<?php

/**
 * Template for the LinkCentral Settings page in the WordPress admin.
 *
 * This template displays various configuration options for the LinkCentral plugin,
 * including general settings, tracking settings, usage instructions, and premium features.
 */
if ( !defined( 'ABSPATH' ) ) {
    exit;
}
// Exit if accessed directly
do_action( 'linkcentral_admin_header' );
?>

<div class="wrap linkcentral-settings">
    <h1><?php 
esc_html_e( 'Settings', 'linkcentral' );
?></h1>
    
    <?php 
settings_errors( 'linkcentral_messages' );
?>

    <?php 
/**
 * Navigation Tabs
 * 
 * Displays tabs for different sections of the settings page.
 */
?>
    <h2 class="nav-tab-wrapper">
        <a href="#linkcentral-general" class="nav-tab <?php 
echo ( $active_tab == 'general' ? 'nav-tab-active' : '' );
?>">
            <span class="dashicons dashicons-admin-generic"></span>
            <?php 
esc_html_e( 'General', 'linkcentral' );
?>
        </a>
        <a href="#linkcentral-tracking" class="nav-tab <?php 
echo ( $active_tab == 'tracking' ? 'nav-tab-active' : '' );
?>">
            <span class="dashicons dashicons-chart-bar"></span>
            <?php 
esc_html_e( 'Tracking', 'linkcentral' );
?>
        </a>
        <a href="#linkcentral-roles" class="nav-tab <?php 
echo ( $active_tab == 'roles' ? 'nav-tab-active' : '' );
?>">
            <span class="dashicons dashicons-admin-users"></span>
            <?php 
esc_html_e( 'Roles', 'linkcentral' );
?>
        </a>
        <a href="#linkcentral-import-export" class="nav-tab <?php 
echo ( $active_tab == 'import-export' ? 'nav-tab-active' : '' );
?>">
            <span class="dashicons dashicons-database-import"></span>
            <?php 
esc_html_e( 'Import/Export', 'linkcentral' );
?>
        </a>
        <a href="#linkcentral-plugin-behaviour" class="nav-tab <?php 
echo ( $active_tab == 'usage' ? 'nav-tab-active' : '' );
?>">
            <span class="dashicons dashicons-admin-plugins"></span>
            <?php 
esc_html_e( 'Plugin', 'linkcentral' );
?>
        </a>
        <a href="#linkcentral-usage" class="nav-tab <?php 
echo ( $active_tab == 'usage' ? 'nav-tab-active' : '' );
?>">
            <span class="dashicons dashicons-book"></span>
            <?php 
esc_html_e( 'Usage', 'linkcentral' );
?>
        </a>
        <a href="#linkcentral-premium" class="nav-tab <?php 
echo ( $active_tab == 'premium' ? 'nav-tab-active' : '' );
?>">
            <span class="dashicons dashicons-star-filled"></span>
            <?php 
esc_html_e( 'Premium', 'linkcentral' );
?>
        </a>
    </h2>

    <form method="post" action="">
        <?php 
wp_nonce_field( 'linkcentral_save_settings', 'linkcentral_settings_nonce' );
?>
        <input type="hidden" name="active_tab" id="active_tab" value="<?php 
echo esc_attr( $active_tab );
?>">
        
        <?php 
/**
 * General Settings Section
 * 
 * Contains settings for URL prefix, global link attributes, and other general options.
 */
?>
        <div id="linkcentral-general" class="linkcentral-tab-content">
            <h3><?php 
esc_html_e( 'Link Settings', 'linkcentral' );
?></h3>
            <table class="linkcentral-form-table form-table">
                <tr>
                    <th scope="row">
                        <label for="linkcentral_url_prefix"><?php 
esc_html_e( 'Custom URL Prefix', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'This is the default URL prefix for all links. You can change it to a prefix of your choice. If you want to disable the prefix for a specific link, you can do so on the link edit page.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <select id="linkcentral_url_prefix_select">
                            <?php 
foreach ( $this->get_preset_prefixes() as $prefix ) {
    ?>
                                <option value="<?php 
    echo esc_attr( $prefix );
    ?>" <?php 
    selected( $url_prefix, $prefix );
    ?>><?php 
    echo esc_html( $prefix );
    ?></option>
                            <?php 
}
?>
                            <option value="custom" <?php 
selected( !in_array( $url_prefix, $this->get_preset_prefixes() ) );
?>><?php 
esc_attr_e( 'Custom:', 'linkcentral' );
?></option>
                        </select>
                        <input type="text" id="linkcentral_url_prefix" name="linkcentral_url_prefix" value="<?php 
echo esc_attr( $url_prefix );
?>" class="regular-text" <?php 
echo ( in_array( $url_prefix, $this->get_preset_prefixes() ) ? 'style="display:none;"' : '' );
?>>
                        <p class="description">
                            <?php 
echo sprintf( 
    // translators: %1$s is the main site URL, %2$s is the custom URL prefix
    esc_html__( 'Your custom URLs look like this: %1$s/%2$s/custom-link.', 'linkcentral' ),
    esc_url( get_site_url() ),
    '<span id="prefix-example">' . esc_html( $url_prefix ) . '</span>'
 );
?>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="linkcentral_default_disable_prefix"><?php 
esc_html_e( 'Disable Prefix by Default', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'When creating new links, this setting determines whether the URL prefix is enabled or disabled by default. You can still toggle this for individual links on the link edit page.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_default_disable_prefix" id="linkcentral_default_disable_prefix" value="1" <?php 
checked( $default_disable_prefix, 1 );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label"><?php 
esc_html_e( 'Disable prefix by default for new links', 'linkcentral' );
?></span>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="linkcentral_case_sensitive_redirects"><?php 
esc_html_e( 'Case-Sensitive Redirects', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'When enabled, link redirects will be case-sensitive. For example, /go/MyLink and /go/mylink will be treated as different links.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_case_sensitive_redirects" id="linkcentral_case_sensitive_redirects" value="1" <?php 
checked( $case_sensitive_redirects, 1 );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label"><?php 
esc_html_e( 'Make redirects case-sensitive', 'linkcentral' );
?></span>
                    </td>
                </tr>
            </table>
            
            <hr>
            <h3><?php 
esc_html_e( 'Global Link Attributes', 'linkcentral' );
?></h3>
            <p class="description"><?php 
esc_html_e( 'The following settings are globally applied to links. They can be overridden for individual links.', 'linkcentral' );
?></p>
            <table class="linkcentral-form-table form-table">
                <tr>
                    <th scope="row">
                        <label for="linkcentral_global_nofollow"><?php 
esc_html_e( 'Global No-Follow Attribute', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'When enabled, all links will have the rel="nofollow" attribute, telling search engines not to follow these links. You can override this for individual links.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_global_nofollow" id="linkcentral_global_nofollow" value="1" <?php 
checked( $global_nofollow, 1 );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label"><?php 
esc_html_e( 'Add no-follow attribute to all links', 'linkcentral' );
?></span>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="linkcentral_global_sponsored"><?php 
esc_html_e( 'Global Sponsored Attribute', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'When enabled, all links will have the rel="sponsored" attribute, indicating that they are paid or sponsored links. You can override this for individual links.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_global_sponsored" id="linkcentral_global_sponsored" value="1" <?php 
checked( $global_sponsored, 1 );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label"><?php 
esc_html_e( 'Add sponsored attribute to all links', 'linkcentral' );
?></span>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="linkcentral_global_redirection_type"><?php 
esc_html_e( 'Global Redirection Type', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'Choose the default redirection type for all links.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <select name="linkcentral_global_redirection_type" id="linkcentral_global_redirection_type">
                            <option value="307" <?php 
selected( $global_redirection_type, '307' );
?>><?php 
esc_html_e( '307 (Temporary)', 'linkcentral' );
?></option>
                            <option value="302" <?php 
selected( $global_redirection_type, '302' );
?>><?php 
esc_html_e( '302 (Temporary)', 'linkcentral' );
?></option>
                            <option value="301" <?php 
selected( $global_redirection_type, '301' );
?>><?php 
esc_html_e( '301 (Permanent)', 'linkcentral' );
?></option>
                        </select>
                    </td>
                </tr>
                <tr class="linkcentral-premium-feature <?php 
echo ( linkcentral_fs()->can_use_premium_code__premium_only() ? 'linkcentral-premium-active' : '' );
?>">
                    <th scope="row">
                        <label for="linkcentral_global_parameter_forwarding"><?php 
esc_html_e( 'Global Parameter Forwarding', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'When enabled, query parameters from the custom URL will be appended to the destination URL. For more info, please see', 'linkcentral' );
?> <a href='https://designforwp.com/docs/linkcentral/getting-started/customize-settings/#global-link-attributes' target='_blank'>the documentation</a>."></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_global_parameter_forwarding" id="linkcentral_global_parameter_forwarding" value="1" <?php 
checked( linkcentral_fs()->can_use_premium_code__premium_only() && $global_parameter_forwarding, 1 );
?> <?php 
echo ( !linkcentral_fs()->can_use_premium_code__premium_only() ? 'disabled' : '' );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label">
                            <?php 
esc_html_e( 'Enable parameter forwarding for all links', 'linkcentral' );
?>
                            <?php 
?>
                                <a href="#linkcentral-premium" class="linkcentral-premium-tag"><?php 
esc_html_e( 'Premium', 'linkcentral' );
?></a>
                            <?php 
?>
                        </span>
                    </td>
                </tr>
                <tr class="linkcentral-premium-feature <?php 
echo ( linkcentral_fs()->can_use_premium_code__premium_only() ? 'linkcentral-premium-active' : '' );
?>">
                    <th scope="row">
                        <label for="linkcentral_custom_css_classes"><?php 
esc_html_e( 'Global CSS Classes', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'Add custom CSS classes to be applied to all link insertions. Separate multiple classes with spaces. Gutenberg Buttons are not supported.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <input type="text" name="linkcentral_custom_css_classes" id="linkcentral_custom_css_classes" value="<?php 
echo esc_attr( $custom_css_classes );
?>" class="regular-text" <?php 
echo ( !linkcentral_fs()->can_use_premium_code__premium_only() ? 'disabled' : '' );
?>>
                        <?php 
?>
                            <a href="#linkcentral-premium" class="linkcentral-premium-tag"><?php 
esc_html_e( 'Premium', 'linkcentral' );
?></a>
                        <?php 
?>
                    </td>
                </tr>
            </table>
        </div>

        <?php 
/**
 * Tracking Settings Section
 * 
 * Contains settings related to click tracking, IP exclusions, and Google Analytics integration.
 */
?>
        <div id="linkcentral-tracking" class="linkcentral-tab-content" style="display:none;">
            <table class="linkcentral-form-table form-table">
                <tr>
                    <th scope="row">
                        <label for="linkcentral_disable_reporting"><?php 
esc_html_e( 'Disable Reporting', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'When checked, no click data will be collected or stored.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_disable_reporting" id="linkcentral_disable_reporting" value="1" <?php 
checked( $disable_reporting, 1 );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label"><?php 
esc_html_e( 'Disable click data reporting', 'linkcentral' );
?></span>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="linkcentral_track_user_agent"><?php 
esc_html_e( 'Track User Agent', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'When checked, the user agent of the user will be recorded with each click. This information is used to display browser icons in the insights.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_track_user_agent" id="linkcentral_track_user_agent" value="1" <?php 
checked( $track_user_agent, 1 );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label"><?php 
esc_html_e( 'Enable User Agent tracking', 'linkcentral' );
?></span>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="linkcentral_track_unique_visitors"><?php 
esc_html_e( 'Track Unique Visitors', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'When enabled, a persistent cookie (30-day expiration) will be set to track unique visitors for more accurate click statistics.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_track_unique_visitors" id="linkcentral_track_unique_visitors" value="1" <?php 
checked( get_option( 'linkcentral_track_unique_visitors', false ), 1 );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label"><?php 
esc_html_e( 'Enable unique visitor tracking', 'linkcentral' );
?></span>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="linkcentral_enable_ga"><?php 
esc_html_e( 'Enable Google Analytics', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'Enable integration with Google Analytics 4.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_enable_ga" id="linkcentral_enable_ga" value="1" <?php 
checked( get_option( 'linkcentral_enable_ga' ), 1 );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label"><?php 
esc_html_e( 'Enable Google Analytics 4 integration', 'linkcentral' );
?></span>
                        <a href="#" class="linkcentral-configure-link" data-target="ga4" data-toggle-rows="#linkcentral_ga_config_table">
                            <?php 
esc_html_e( 'Configure', 'linkcentral' );
?>
                            <span class="chevron"></span>
                        </a>
                    </td>
                </tr>
                <tr class="linkcentral-configure-row" id="linkcentral_ga_config_table" data-parent="ga4">
                    <td colspan="3">
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="linkcentral_ga_measurement_id"><?php 
esc_html_e( 'Google Analytics 4 Measurement ID', 'linkcentral' );
?></label>
                                </th>
                                <td class="linkcentral-info-icon-cell">
                                    <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'Enter your Google Analytics 4 Measurement ID (starts with G-).', 'linkcentral' );
?>"></span>
                                </td>
                                <td>
                                    <input type="text" id="linkcentral_ga_measurement_id" name="linkcentral_ga_measurement_id" value="<?php 
echo esc_attr( get_option( 'linkcentral_ga_measurement_id' ) );
?>" placeholder="G-XXXXXXXXXX" class="regular-text">
                                    <a href="https://designforwp.com/docs/linkcentral/tracking/set-up-google-analytics-4/" title="More about Google Analytics 4 integration" target="_blank"><?php 
esc_html_e( 'See instructions', 'linkcentral' );
?></a>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="linkcentral_ga_api_secret"><?php 
esc_html_e( 'Google Analytics 4 API Secret', 'linkcentral' );
?></label>
                                </th>
                                <td class="linkcentral-info-icon-cell">
                                    <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'Enter your Google Analytics 4 API Secret.', 'linkcentral' );
?>"></span>
                                </td>
                                <td>
                                    <input type="text" id="linkcentral_ga_api_secret" name="linkcentral_ga_api_secret" value="<?php 
echo esc_attr( get_option( 'linkcentral_ga_api_secret' ) );
?>" class="regular-text">
                                </td>
                            </tr>
                        </table>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="linkcentral_excluded_ips"><?php 
esc_html_e( 'Excluded IP Addresses', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'Enter IP addresses to exclude from tracking, one per line.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <textarea id="linkcentral_excluded_ips" name="linkcentral_excluded_ips" rows="5" cols="50"><?php 
echo esc_textarea( get_option( 'linkcentral_excluded_ips', '' ) );
?></textarea>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="linkcentral_excluded_roles"><?php 
esc_html_e( 'Excluded User Roles', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'Select user roles to exclude from tracking. If someone is signed in as a user with one of these roles, their clicks will not be recorded.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <?php 
$roles = wp_roles()->get_names();
foreach ( $roles as $role_slug => $role_name ) {
    $checked = ( in_array( $role_slug, $excluded_roles ) ? 'checked' : '' );
    echo '<label><input type="checkbox" name="linkcentral_excluded_roles[]" value="' . esc_attr( $role_slug ) . '" ' . esc_attr( $checked ) . '> ' . esc_html( translate_user_role( $role_name ) ) . '</label><br>';
}
?>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="linkcentral_exclude_bots"><?php 
esc_html_e( 'Exclude Bots', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'When checked, clicks from known bots will not be recorded.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_exclude_bots" id="linkcentral_exclude_bots" value="1" <?php 
checked( $exclude_bots, 1 );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label"><?php 
esc_html_e( 'Exclude bots from click tracking', 'linkcentral' );
?></span>
                    </td>
                </tr>
            </table>

            <?php 
?>
            
            <hr>
            <h3><?php 
esc_html_e( 'Deletion', 'linkcentral' );
?></h3>
            <table class="linkcentral-form-table form-table">
                <tr>
                    <th scope="row">
                        <label for="linkcentral_delete_tracking_data_on_link_deletion"><?php 
esc_html_e( 'Delete Tracking Data on Link Deletion', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'If enabled, all associated click tracking data for a link will be permanently deleted when that link is deleted from the system.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_delete_tracking_data_on_link_deletion" id="linkcentral_delete_tracking_data_on_link_deletion" value="1" <?php 
checked( $delete_tracking_data_on_link_deletion, 1 );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label"><?php 
esc_html_e( 'Delete click tracking data when the corresponding link is permanently deleted', 'linkcentral' );
?></span>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="linkcentral_enable_data_expiry"><?php 
esc_html_e( 'Enable Data Expiry', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'If enabled, tracking data older than the specified number of days will be automatically deleted.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_enable_data_expiry" id="linkcentral_enable_data_expiry" value="1" <?php 
checked( $enable_data_expiry, 1 );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label"><?php 
esc_html_e( 'Enable automatic deletion of old tracking data', 'linkcentral' );
?>:</span>
                        <select name="linkcentral_data_expiry_days" id="linkcentral_data_expiry_days" <?php 
disabled( !$enable_data_expiry );
?>>
                            <option value="90" <?php 
selected( $data_expiry_days, 90 );
?>><?php 
esc_html_e( '90 days', 'linkcentral' );
?></option>
                            <option value="180" <?php 
selected( $data_expiry_days, 180 );
?>><?php 
esc_html_e( '180 days', 'linkcentral' );
?></option>
                            <option value="365" <?php 
selected( $data_expiry_days, 365 );
?>><?php 
esc_html_e( '365 days', 'linkcentral' );
?></option>
                        </select>
                    </td>
                </tr>
            </table>
        </div>

        <?php 
/**
 * Roles Settings Section
 * 
 * Contains settings for role capabilities.
 */
?>
        <div id="linkcentral-roles" class="linkcentral-tab-content" style="display:none;">
            <h3><?php 
esc_html_e( 'Role Capabilities', 'linkcentral' );
?></h3>
            <p class="description"><?php 
esc_html_e( 'Assign capabilities to different user roles to control access to LinkCentral functionality.', 'linkcentral' );
?></p>
            <p class="description"><em><?php 
esc_html_e( 'Note: Administrators always have all capabilities and cannot be modified.', 'linkcentral' );
?></em></p>
            
            <?php 
// Get WordPress roles
global $wp_roles;
if ( !isset( $wp_roles ) ) {
    $wp_roles = new WP_Roles();
}
// Define capabilities with descriptions
$capabilities = array(
    'linkcentral_view_links'        => array(
        'label'       => __( 'View Links', 'linkcentral' ),
        'description' => __( 'Can view existing links and insert those into content', 'linkcentral' ),
    ),
    'linkcentral_create_edit_links' => array(
        'label'       => __( 'Create & Edit Links', 'linkcentral' ),
        'description' => __( 'Can create new links and edit existing links', 'linkcentral' ),
    ),
    'linkcentral_view_stats'        => array(
        'label'       => __( 'View Stats', 'linkcentral' ),
        'description' => __( 'Access to view analytics and statistics', 'linkcentral' ),
    ),
    'linkcentral_manage_settings'   => array(
        'label'       => __( 'Manage Settings', 'linkcentral' ),
        'description' => __( 'Access to view and change plugin settings', 'linkcentral' ),
    ),
);
// Get all roles including administrator
$roles = $wp_roles->role_names;
// For each capability, create a card
foreach ( $capabilities as $cap => $cap_info ) {
    ?>
            <hr>
            <div class="linkcentral-capability">
                <div class="linkcentral-capability-header">
                    <h4><?php 
    echo esc_html( $cap_info['label'] );
    ?></h4>
                    <p class="description"><?php 
    echo esc_html( $cap_info['description'] );
    ?></p>
                </div>
                <div class="linkcentral-capability-roles-grid">
                    <?php 
    foreach ( $roles as $role_name => $role_display_name ) {
        // Check if this is the administrator role or editor role for view links capability
        $is_admin = $role_name === 'administrator';
        $is_protected = $is_admin || $role_name === 'editor' && $cap === 'linkcentral_view_links';
        // Get the WP_Role object
        $role_obj = get_role( $role_name );
        // Check if the role has this capability
        $has_cap = $role_obj && $role_obj->has_cap( $cap );
        // Get tooltip text for editor role in view links capability
        $tooltip_text = '';
        if ( $role_name === 'editor' && $cap === 'linkcentral_view_links' ) {
            $tooltip_text = esc_attr__( 'Editors can access all posts and pages, including those created by others, which may contain LinkCentral links. This is why the "View Links" capability is always enabled for editors and cannot be changed.', 'linkcentral' );
        }
        ?>
                    <div class="linkcentral-capability-role-item <?php 
        echo ( $is_protected ? 'linkcentral-capability-role-protected' : '' );
        ?>">
                        <label class="linkcentral-capability-role-label">
                            <span>
                                <span class="linkcentral-capability-role-name"><?php 
        echo esc_html( translate_user_role( $role_display_name ) );
        ?></span>
                                <?php 
        if ( $tooltip_text ) {
            ?>
                                    <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
            echo $tooltip_text;
            ?>"></span>
                                <?php 
        }
        ?>
                            </span>
                            <span class="linkcentral-toggle-switch">
                                <input type="checkbox" 
                                       name="<?php 
        echo ( $is_protected ? '' : 'linkcentral_capabilities[' . esc_attr( $role_name ) . '][]' );
        ?>" 
                                       value="<?php 
        echo esc_attr( $cap );
        ?>"
                                       <?php 
        checked( $has_cap, true );
        ?>
                                       <?php 
        disabled( $is_protected, true );
        ?>>
                                <span class="linkcentral-toggle-slider"></span>
                            </span>
                        </label>
                    </div>
                    <?php 
    }
    ?>
                </div>
            </div>
            <?php 
}
?>
        </div>

        <?php 
/**
 * Import/Export Settings Section
 * 
 * Contains tools for importing links and exporting LinkCentral data
 */
?>
        <div id="linkcentral-import-export" class="linkcentral-tab-content" style="display:none;">
            <h3><?php 
esc_html_e( 'Import Links', 'linkcentral' );
?></h3>
            <p><?php 
esc_html_e( 'Import data from your LinkCentral Export or from other plugins.', 'linkcentral' );
?></p>
            <p><?php 
esc_html_e( 'LinkCentral can import links from BetterLinks, Pretty Links, ThirstyAffiliates, and any other CSV file.', 'linkcentral' );
?></p>
            
            <div class="linkcentral-import-grid">
                <div class="linkcentral-import-card">
                    <h4>
                        <span class="dashicons dashicons-media-spreadsheet"></span>
                        <?php 
esc_html_e( 'Import from CSV', 'linkcentral' );
?>
                    </h4>
                    <p><?php 
esc_html_e( 'Import links from a CSV file.', 'linkcentral' );
?></p>
                    <a href="<?php 
echo admin_url( 'admin.php?page=linkcentral-csv-import' );
?>" class="button button-primary">
                        <?php 
esc_html_e( 'Import CSV File', 'linkcentral' );
?>
                    </a>
                </div>

                <div class="linkcentral-import-card">
                    <h4>
                        <span class="dashicons dashicons-database-export"></span>
                        <?php 
esc_html_e( 'Import from JSON', 'linkcentral' );
?>
                    </h4>
                    <p><?php 
esc_html_e( 'Restore or migrate LinkCentral data from JSON export.', 'linkcentral' );
?></p>
                    <a href="<?php 
echo admin_url( 'admin.php?page=linkcentral-json-import' );
?>" class="button button-primary">
                        <?php 
esc_html_e( 'Import JSON Export', 'linkcentral' );
?>
                    </a>
                </div>

                <div class="linkcentral-import-card linkcentral-popular-plugins">
                    <h4>
                        <span class="dashicons dashicons-admin-plugins"></span>
                        <?php 
esc_html_e( 'Migrate from Popular Plugins', 'linkcentral' );
?>
                    </h4>
                    <p><?php 
esc_html_e( 'Import links, categories, and analytics from other popular WordPress plugins.', 'linkcentral' );
?></p>
                    <div class="linkcentral-plugin-buttons">
                        <a href="<?php 
echo admin_url( 'admin.php?page=linkcentral-thirstyaffiliates-import' );
?>" class="button button-primary">
                            <?php 
esc_html_e( 'Import from ThirstyAffiliates', 'linkcentral' );
?>
                        </a>
                        <a href="<?php 
echo admin_url( 'admin.php?page=linkcentral-prettylinks-import' );
?>" class="button button-primary">
                            <?php 
esc_html_e( 'Import from Pretty Links', 'linkcentral' );
?>
                        </a>
                    </div>
                </div>
            </div>

            <hr>

            <div class="linkcentral-export-section">
                <h3><?php 
esc_html_e( 'Export Data', 'linkcentral' );
?></h3>
                <p class="description"><?php 
esc_html_e( 'Export your LinkCentral links and analytics data to JSON format.', 'linkcentral' );
?></p>
                
                <?php 
?>
                    <button type="button" id="linkcentral-start-export" class="button button-secondary" disabled>
                        <?php 
esc_html_e( 'Export Data', 'linkcentral' );
?>
                    </button>
                    <a href="<?php 
echo esc_url( admin_url( 'admin.php?page=linkcentral-settings#linkcentral-premium' ) );
?>" class="linkcentral-premium-tag"><?php 
esc_html_e( 'Premium', 'linkcentral' );
?></a>
                <?php 
?>
            </div>
        </div>

        <?php 
/**
 * Plugin Behaviour Settings Section
 * 
 * Contains settings related to plugin behaviour
 */
?>
        <div id="linkcentral-plugin-behaviour" class="linkcentral-tab-content" style="display:none;">
            <h3><?php 
esc_html_e( 'Plugin behaviour', 'linkcentral' );
?></h3>
            <table class="linkcentral-form-table form-table">
                <tr>
                    <th scope="row">
                        <label for="linkcentral_default_link_insertion_type"><?php 
esc_html_e( 'Default Insertion Type', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'Specify the default link insertion method to be pre-selected when adding new links into your website using Gutenberg or TinyMCE.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <select name="linkcentral_default_link_insertion_type" id="linkcentral_default_link_insertion_type">
                            <option value="synchronized" <?php 
selected( get_option( 'linkcentral_default_link_insertion_type', 'synchronized' ), 'synchronized' );
?>><?php 
esc_html_e( 'Synchronized', 'linkcentral' );
?></option>
                            <option value="direct" <?php 
selected( get_option( 'linkcentral_default_link_insertion_type' ), 'direct' );
?>><?php 
esc_html_e( 'Direct', 'linkcentral' );
?></option>
                            <option value="shortcode" <?php 
selected( get_option( 'linkcentral_default_link_insertion_type' ), 'shortcode' );
?>><?php 
esc_html_e( 'Shortcode', 'linkcentral' );
?></option>
                        </select>
                    </td>
                </tr>
            </table>

            <hr>

            <h3><?php 
esc_html_e( 'Additional Features', 'linkcentral' );
?></h3>
            <table class="linkcentral-form-table form-table">
                <tr>
                    <th scope="row">
                        <label for="linkcentral_enable_qr_codes"><?php 
esc_html_e( 'Enable QR Code Generation', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'This adds a button in the Tools section on the Add/Edit link page to generate QR codes for the short link.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_enable_qr_codes" id="linkcentral_enable_qr_codes" value="1" <?php 
checked( get_option( 'linkcentral_enable_qr_codes', false ), 1 );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label"><?php 
esc_html_e( 'Enable QR code generation for links', 'linkcentral' );
?></span>
                        <a href="#" class="linkcentral-configure-link" data-target="qr-codes" data-toggle-rows="#linkcentral_qr_codes_config_table">
                            <?php 
esc_html_e( 'Configure', 'linkcentral' );
?>
                            <span class="chevron"></span>
                        </a>
                    </td>
                </tr>
                <tr class="linkcentral-configure-row" id="linkcentral_qr_codes_config_table" data-parent="qr-codes">
                    <td colspan="3">
                        <?php 
include LINKCENTRAL_PLUGIN_DIR . 'views/settings/qr-code-config.php';
?>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="linkcentral_enable_dashboard_widget"><?php 
esc_html_e( 'Enable Dashboard Widget', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'Enable the LinkCentral dashboard widget to display click analytics on your WordPress dashboard. Shows 24-hour and configurable period statistics for your links.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_enable_dashboard_widget" id="linkcentral_enable_dashboard_widget" value="1" <?php 
checked( get_option( 'linkcentral_enable_dashboard_widget', false ), 1 );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label"><?php 
esc_html_e( 'Show LinkCentral analytics on dashboard', 'linkcentral' );
?></span>
                        <a href="#" class="linkcentral-configure-link" data-target="dashboard-widget" data-toggle-rows="#linkcentral_dashboard_widget_config_table">
                            <?php 
esc_html_e( 'Configure', 'linkcentral' );
?>
                            <span class="chevron"></span>
                        </a>
                    </td>
                </tr>
                <tr class="linkcentral-configure-row" id="linkcentral_dashboard_widget_config_table" data-parent="dashboard-widget">
                    <td colspan="3">
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="linkcentral_dashboard_widget_period"><?php 
esc_html_e( 'Analytics Period', 'linkcentral' );
?></label>
                                </th>
                                <td class="linkcentral-info-icon-cell">
                                    <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'Choose another time period (in addition to 24 hours) to show click analytics in the dashboard widget.', 'linkcentral' );
?>"></span>
                                <td>
                                    <select name="linkcentral_dashboard_widget_period" id="linkcentral_dashboard_widget_period">
                                        <option value="7" <?php 
selected( get_option( 'linkcentral_dashboard_widget_period', '7' ), '7' );
?>><?php 
esc_html_e( '7 days', 'linkcentral' );
?></option>
                                        <option value="14" <?php 
selected( get_option( 'linkcentral_dashboard_widget_period', '7' ), '14' );
?>><?php 
esc_html_e( '14 days', 'linkcentral' );
?></option>
                                        <option value="30" <?php 
selected( get_option( 'linkcentral_dashboard_widget_period', '7' ), '30' );
?>><?php 
esc_html_e( '30 days', 'linkcentral' );
?></option>
                                    </select>
                                </td>
                            </tr>
                        </table>
                    </td>
                </tr>
                <tr class="linkcentral-premium-feature <?php 
echo ( linkcentral_fs()->can_use_premium_code__premium_only() ? 'linkcentral-premium-active' : '' );
?>">
                    <th scope="row">
                        <label for="linkcentral_enable_utm_templates"><?php 
esc_html_e( 'Enable UTM Templates', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'Enable UTM Templates to quickly apply predefined UTM parameters to your links. This will show up a small UTM button on the link creation page next to the Destination URL field to quickly apply UTM parameters to a link.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_enable_utm_templates" id="linkcentral_enable_utm_templates" value="1" <?php 
checked( get_option( 'linkcentral_enable_utm_templates' ), 1 );
?> <?php 
echo ( !linkcentral_fs()->can_use_premium_code__premium_only() ? 'disabled' : '' );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label"><?php 
esc_html_e( 'Enable UTM Templates', 'linkcentral' );
?></span>
                        <?php 
?>
                            <a href="#linkcentral-premium" class="linkcentral-premium-tag"><?php 
esc_html_e( 'Premium', 'linkcentral' );
?></a>
                        <?php 
?>
                    </td>
                </tr>
                <?php 
?>
                <tr class="linkcentral-premium-feature <?php 
echo ( linkcentral_fs()->can_use_premium_code__premium_only() ? 'linkcentral-premium-active' : '' );
?>">
                    <th scope="row">
                        <label for="linkcentral_enable_broken_links_checker"><?php 
esc_html_e( 'Enable Broken Links Checker', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'Enable the broken links checker to automatically detect broken destination URLs in your links. This feature runs daily or weekly and can send email notifications when broken links are found.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_enable_broken_links_checker" id="linkcentral_enable_broken_links_checker" value="1" <?php 
checked( get_option( 'linkcentral_enable_broken_links_checker', false ) );
?> <?php 
echo ( !linkcentral_fs()->can_use_premium_code__premium_only() ? 'disabled' : '' );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label">
                            <?php 
esc_html_e( 'Enable broken links checker', 'linkcentral' );
?>
                            <?php 
?>
                                <a href="#linkcentral-premium" class="linkcentral-premium-tag"><?php 
esc_html_e( 'Premium', 'linkcentral' );
?></a>
                            <?php 
?>
                        </span>
                    </td>
                </tr>
                <?php 
?>
                <tr class="linkcentral-premium-feature <?php 
echo ( linkcentral_fs()->can_use_premium_code__premium_only() ? 'linkcentral-premium-active' : '' );
?>">
                    <th scope="row">
                        <label for="linkcentral_enable_webhooks"><?php 
esc_html_e( 'Enable Webhooks', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'Enable webhooks to send HTTP requests to external services when links are clicked. Webhooks are sent asynchronously and do not delay redirections.', 'linkcentral' );
?> <a href='https://designforwp.com/docs/linkcentral/tools-and-other/webhooks/' target='_blank'><?php 
esc_html_e( 'Learn more', 'linkcentral' );
?></a>"></span>
                    </td>
                    <td>
                        <label class="linkcentral-toggle-switch">
                            <input type="checkbox" name="linkcentral_enable_webhooks" id="linkcentral_enable_webhooks" value="1" <?php 
checked( get_option( 'linkcentral_enable_webhooks', false ) );
?> <?php 
echo ( !linkcentral_fs()->can_use_premium_code__premium_only() ? 'disabled' : '' );
?>>
                            <span class="linkcentral-toggle-slider"></span>
                        </label>
                        <span class="linkcentral-toggle-label">
                            <?php 
esc_html_e( 'Enable webhooks on link clicks', 'linkcentral' );
?>
                            <?php 
?>
                                <a href="#linkcentral-premium" class="linkcentral-premium-tag"><?php 
esc_html_e( 'Premium', 'linkcentral' );
?></a>
                            <?php 
?>
                        </span>
                    </td>
                </tr>
                <?php 
?>
                <?php 
?>
                    <tr class="linkcentral-premium-feature">
                        <th scope="row">
                            <label for="linkcentral_enable_auto_keyword_linking"><?php 
esc_html_e( 'Enable Auto-Keyword Linking', 'linkcentral' );
?></label>
                        </th>
                        <td class="linkcentral-info-icon-cell">
                            <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'Automatically replace keywords in your content with LinkCentral links. Define keywords for each link and set replacement density levels.', 'linkcentral' );
?>"></span>
                        </td>
                        <td>
                            <label class="linkcentral-toggle-switch">
                                <input type="checkbox" name="linkcentral_enable_auto_keyword_linking" id="linkcentral_enable_auto_keyword_linking" value="1" disabled>
                                <span class="linkcentral-toggle-slider"></span>
                            </label>
                            <span class="linkcentral-toggle-label">
                                <?php 
esc_html_e( 'Enable auto-keyword linking', 'linkcentral' );
?>
                                <a href="#linkcentral-premium" class="linkcentral-premium-tag"><?php 
esc_html_e( 'Premium', 'linkcentral' );
?></a>
                            </span>
                        </td>
                    </tr>
                <?php 
?>
            </table>

            <hr>
            <h3><?php 
esc_html_e( 'Plugin Data Reset', 'linkcentral' );
?></h3>
            <table class="linkcentral-form-table form-table">
                <tr>
                    <th scope="row">
                        <label><?php 
esc_html_e( 'Data Reset Options', 'linkcentral' );
?></label>
                    </th>
                    <td class="linkcentral-info-icon-cell">
                        <span class="linkcentral-info-icon dashicons dashicons-info-outline" data-tooltip="<?php 
esc_attr_e( 'Permanently delete LinkCentral data. These actions cannot be undone.', 'linkcentral' );
?>"></span>
                    </td>
                    <td>
                        <span><?php 
esc_html_e( 'Use these options to reset your LinkCentral data.', 'linkcentral' );
?></span>
                        <a href="#" class="linkcentral-configure-link" data-target="data-reset" data-toggle-rows="#linkcentral_data_reset_config_table">
                            <?php 
esc_html_e( 'Continue', 'linkcentral' );
?>
                            <span class="chevron"></span>
                        </a>
                    </td>
                </tr>
                <tr class="linkcentral-configure-row" id="linkcentral_data_reset_config_table" data-parent="data-reset">
                    <td colspan="3">
                        <div class="linkcentral-data-reset-section">
                            <div class="linkcentral-reset-buttons">
                                <button type="button" name="linkcentral_reset_clicks" class="button button-secondary" style="color: #d63638; border-color: #d63638; margin-right: 10px;" onclick="if (confirm('<?php 
esc_attr_e( 'Are you sure you want to reset all clicking data? This action cannot be undone.', 'linkcentral' );
?>')) { this.type='submit'; this.form.submit(); }">
                                    <?php 
esc_html_e( 'Reset all clicking data', 'linkcentral' );
?>
                                </button>
                                <button type="button" name="linkcentral_reset_all" class="button button-secondary" style="color: #fff; background-color: #d63638; border-color: #d63638;" onclick="if (confirm('<?php 
esc_attr_e( 'Are you sure you want to reset ALL LinkCentral data? This will permanently delete all links, click data, and categories. This action cannot be undone.', 'linkcentral' );
?>')) { this.type='submit'; this.form.submit(); }">
                                    <?php 
esc_html_e( 'Reset ALL data', 'linkcentral' );
?>
                                </button>
                            </div>
                        </div>
                    </td>
                </tr>
            </table>
        </div>


        <?php 
/**
 * Usage Instructions Section
 * 
 * Provides information on how to use LinkCentral shortcodes and integrate with various editors.
 */
?>
        <div id="linkcentral-usage" class="linkcentral-tab-content" style="display:none;">
            <h3><?php 
esc_html_e( 'How to Use LinkCentral', 'linkcentral' );
?></h3>
            
            <p><?php 
esc_html_e( 'There are two ways to add LinkCentral links to your content:', 'linkcentral' );
?></p>

            <div id="linkcentral-usage-methods">
                <!-- Recommended Method -->
                <div class="linkcentral-usage-method linkcentral-usage-method-recommended">
                    <h4>
                        <span class="dashicons dashicons-yes-alt"></span>
                        <?php 
esc_html_e( 'Recommended: Using Integrations', 'linkcentral' );
?>
                    </h4>
                    <div class="linkcentral-usage-intro">
                        <p><?php 
esc_html_e( 'Use our built-in integrations to insert links that automatically sync when changes are made. When you update a link in LinkCentral, all integrated links update automatically across your site.', 'linkcentral' );
?></p>
                        <a href="https://designforwp.com/docs/linkcentral/creating-and-using-links/inserting-links-into-your-content/" target="_blank" class="button button-secondary">
                            <?php 
esc_html_e( 'Learn More in the Documentation', 'linkcentral' );
?>
                        </a>
                    </div>

                    <div class="linkcentral-accordion-container">
                        <div class="linkcentral-accordion">
                            <h4 class="linkcentral-accordion-header">1. <?php 
esc_html_e( 'Shortcodes', 'linkcentral' );
?></h4>
                            <div class="linkcentral-accordion-content">
                                <p><?php 
esc_html_e( 'Use shortcodes to insert LinkCentral links anywhere in your content:', 'linkcentral' );
?></p>
                                <code>[linkcentral id="123"]Your Link Text[/linkcentral]</code>
                                <p><strong><?php 
esc_html_e( 'Options:', 'linkcentral' );
?></strong></p>
                                <ul>
                                    <li>id="{id}" <?php 
esc_html_e( '(required)', 'linkcentral' );
?></li>
                                    <li>newtab="true"</li>
                                    <li>parameters="p1=v1&p2=v2" (<?php 
esc_html_e( 'Premium only', 'linkcentral' );
?>)</li>
                                </ul>
                            </div>
                        </div>

                        <div class="linkcentral-accordion">
                            <h4 class="linkcentral-accordion-header">2. <?php 
esc_html_e( 'Gutenberg (Block Editor)', 'linkcentral' );
?></h4>
                            <div class="linkcentral-accordion-content">
                                <p><?php 
esc_html_e( 'In the Gutenberg editor, use the LinkCentral button in the toolbar:', 'linkcentral' );
?></p>
                                <ol>
                                    <li><?php 
esc_html_e( 'Select text in a Paragraph block or Button block', 'linkcentral' );
?></li>
                                    <li><?php 
esc_html_e( 'Click the LinkCentral icon in the toolbar', 'linkcentral' );
?></li>
                                    <li><?php 
esc_html_e( 'Search for and select your link', 'linkcentral' );
?></li>
                                </ol>
                            </div>
                        </div>

                        <div class="linkcentral-accordion">
                            <h4 class="linkcentral-accordion-header">3. <?php 
esc_html_e( 'Classic Editor (TinyMCE)', 'linkcentral' );
?></h4>
                            <div class="linkcentral-accordion-content">
                                <p><?php 
esc_html_e( 'In the Classic Editor, use the LinkCentral button in the toolbar:', 'linkcentral' );
?></p>
                                <ol>
                                    <li><?php 
esc_html_e( 'Select text or place cursor where you want to insert the link', 'linkcentral' );
?></li>
                                    <li><?php 
esc_html_e( 'Click the LinkCentral button in the editor toolbar', 'linkcentral' );
?></li>
                                    <li><?php 
esc_html_e( 'Search for and select your link, then click "Insert"', 'linkcentral' );
?></li>
                                </ol>
                            </div>
                        </div>

                        <div class="linkcentral-accordion">
                            <h4 class="linkcentral-accordion-header">4. <?php 
esc_html_e( 'Elementor', 'linkcentral' );
?></h4>
                            <div class="linkcentral-accordion-content">
                                <p><?php 
esc_html_e( 'Elementor uses the Classic Editor for content. See the steps above to use this integration.', 'linkcentral' );
?></p>
                                <p><?php 
esc_html_e( 'LinkCentral also integrates with Elementor Pro\'s dynamic tags:', 'linkcentral' );
?></p>
                                <ol>
                                    <li><?php 
esc_html_e( 'Edit any element that supports URL input', 'linkcentral' );
?></li>
                                    <li><?php 
esc_html_e( 'Click the dynamic tag icon next to the URL field', 'linkcentral' );
?></li>
                                    <li><?php 
esc_html_e( 'Select "LinkCentral Link" from the list', 'linkcentral' );
?></li>
                                    <li><?php 
esc_html_e( 'Choose your LinkCentral link from the dropdown', 'linkcentral' );
?></li>
                                </ol>
                            </div>
                        </div>

                        <div class="linkcentral-accordion">
                            <h4 class="linkcentral-accordion-header">5. <?php 
esc_html_e( 'Beaver Builder', 'linkcentral' );
?></h4>
                            <div class="linkcentral-accordion-content">
                                <p><?php 
esc_html_e( 'Use LinkCentral with Beaver Builder\'s text editor:', 'linkcentral' );
?></p>
                                <ol>
                                    <li><?php 
esc_html_e( 'Edit a text module', 'linkcentral' );
?></li>
                                    <li><?php 
esc_html_e( 'In the text editor, use the LinkCentral button (similar to Classic Editor)', 'linkcentral' );
?></li>
                                    <li><?php 
esc_html_e( 'Search for and select your link, then click "Insert"', 'linkcentral' );
?></li>
                                </ol>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Manual Method -->
                <div class="linkcentral-usage-method linkcentral-usage-method-manual">
                    <h4>
                        <span class="dashicons dashicons-info"></span>
                        <?php 
esc_html_e( 'Alternative: Manual Insertion', 'linkcentral' );
?>
                    </h4>
                    <p><?php 
esc_html_e( 'You can manually copy and paste LinkCentral URLs anywhere in your content. However, these links won\'t automatically update their attributes or slug when changes are made in LinkCentral.', 'linkcentral' );
?></p>
                </div>
            </div>
        </div>

        <?php 
/**
 * Premium Features Section
 * 
 * Displays information about premium features and license management.
 */
?>
        <div id="linkcentral-premium" class="linkcentral-tab-content" style="display:none;">
            <div>
                <?php 
?>
                    <div id="linkcentral-premium-upselling-container">
                        <div id="linkcentral-premium-upselling-bg-shapes">
                            <div class="linkcentral-premium-upselling-bg-shapes_soft-shape-1"></div>
                            <div class="linkcentral-premium-upselling-bg-shapes_soft-shape-2"></div>
                        </div>
                        <div id="linkcentral-premium-upselling-card">
                            <div class="linkcentral-premium-header">
                                <div class="linkcentral-premium-header-text">
                                    <h3><?php 
esc_html_e( 'LinkCentral Premium', 'linkcentral' );
?></h3>
                                    <p><?php 
esc_html_e( 'Unlock access to premium features.', 'linkcentral' );
?></p>
                                </div>
                                <img src="<?php 
echo esc_url( LINKCENTRAL_PLUGIN_URL . 'assets/images/linkcentral-premium-logo.svg' );
?>" width="50" alt="LinkCentral Premium Logo">
                            </div>
                            <hr>
                            <ul>
                                <li><?php 
esc_html_e( 'Advanced and link-specific insights', 'linkcentral' );
?></li>
                                <li><?php 
esc_html_e( 'Dynamic redirects', 'linkcentral' );
?></li>
                                <li><?php 
esc_html_e( 'Parameter forwarding', 'linkcentral' );
?></li>
                                <li><?php 
esc_html_e( 'Custom styling with CSS classes', 'linkcentral' );
?></li>
                                <li><?php 
esc_html_e( 'Broken Links Checker', 'linkcentral' );
?></li>
                                <li><?php 
esc_html_e( 'Webhooks', 'linkcentral' );
?></li>
                                <li><?php 
esc_html_e( 'UTM templates', 'linkcentral' );
?></li>
                                <li><?php 
esc_html_e( 'Auto-Keyword Linking', 'linkcentral' );
?></li>
                                <li><?php 
esc_html_e( 'Split testing, data export, and much more!', 'linkcentral' );
?></li>
                                <li><a href="https://designforwp.com/product/linkcentral-premium/" target="_blank" rel="noopener noreferrer"><?php 
esc_html_e( 'See all features', 'linkcentral' );
?></a></li>
                            </ul>
                            <hr>
                            <a class="linkcentral-buy-now" href="https://designforwp.com/product/linkcentral-premium/" target="_blank" rel="noopener noreferrer"><?php 
esc_html_e( 'Get Premium', 'linkcentral' );
?></a>
                            <p class="linkcentral-money-back-guarantee"><?php 
esc_html_e( '14-Day Money-Back Guarantee', 'linkcentral' );
?></p>
                        </div>
                    </div>
                <?php 
?>
            </div>
        </div>

        <?php 
submit_button();
?>
    </form>

</div>
