<?php
/**
 * Generic Import Page Template
 *
 * This template displays a generic import interface that can be used for any import type.
 * Required variables:
 * - $import_config: Array with import type configuration
 * - $is_plugin_installed: Boolean indicating if the source plugin is installed
 * - $stats: Array with statistics about available data
 */
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// Extract config for easier access
$import_type = $import_config['type'] ?? 'unknown';
$plugin_name = $import_config['plugin_name'] ?? 'Unknown Plugin';
$stats_config = $import_config['stats'] ?? array();
$form_config = $import_config['form'] ?? array();
?>

<div class="wrap linkcentral-import">
    <div class="linkcentral-content">
        <div class="linkcentral-page-header">
            <h1>
                <?php
                /* translators: %s: Plugin name (e.g., Super URL Shortener) */
                printf(esc_html__('Import from %s', 'linkcentral'), $plugin_name);
                ?>
            </h1>
        </div>

        <?php if (!$is_plugin_installed): ?>
            <div class="notice notice-warning">
                <p>
                    <strong>
                        <?php
                        /* translators: %s: Plugin name (e.g., Super URL Shortener) */
                        printf(esc_html__('%s Not Found', 'linkcentral'), $plugin_name);
                        ?>
                    </strong><br>
                    <?php
                    /* translators: 1: Plugin name, 2: Plugin name */
                    printf(esc_html__('No %1$s data was found in your database. Make sure %2$s is installed, activated, and you have created some data.', 'linkcentral'), $plugin_name, $plugin_name);
                    ?>
                </p>
                <p>
                    <a href="<?php echo admin_url('admin.php?page=linkcentral-settings#linkcentral-import-export'); ?>" class="button">
                        <?php esc_html_e('Back to Import Options', 'linkcentral'); ?>
                    </a>
                </p>
            </div>
        <?php else: ?>
            <div class="linkcentral-card">
                <h2>
                    <?php
                    /* translators: %s: Plugin name (e.g., Super URL Shortener) */
                    printf(esc_html__('%s Data Found', 'linkcentral'), $plugin_name);
                    ?>
                </h2>

                <div class="linkcentral-import-stats">
                    <?php foreach ($stats_config as $stat_key => $stat_config): ?>
                        <div class="linkcentral-stat-item">
                            <div class="linkcentral-stat-number"><?php echo number_format($stats[$stat_key] ?? 0); ?></div>
                            <div class="linkcentral-stat-label"><?php echo esc_html($stat_config['label']); ?></div>
                        </div>
                    <?php endforeach; ?>
                </div>

                <form id="linkcentral-import-form" enctype="multipart/form-data">
                    <h3><?php esc_html_e('Import Options', 'linkcentral'); ?></h3>

                    <table class="linkcentral-form-table form-table">
                        <?php foreach ($form_config as $field_key => $field_config): ?>
                            <tr>
                                <th scope="row">
                                    <label for="linkcentral-<?php echo esc_attr($field_key); ?>"><?php echo esc_html($field_config['label']); ?></label>
                                </th>
                                <td>
                                    <?php
                                    // Map field keys to stat keys for checking availability
                                    $stat_key_map = array(
                                        'import_categories' => 'categories',
                                        'import_stats' => 'stats',
                                        'import_clicks' => 'clicks',
                                        'file_upload' => 'files',
                                        'import_mode' => 'exports',
                                        // Add more mappings as needed
                                    );

                                    $stat_key = $stat_key_map[$field_key] ?? $field_key;
                                    $stat_count = $stats[$stat_key] ?? 0;
                                    $is_available = $stat_count > 0;

                                    // Special handling for prefix update field - always available if shown
                                    if ($field_key === 'update_url_prefix') {
                                        $is_available = true;
                                    }

                                    // Replace {count} placeholder with actual count
                                    $description = str_replace('{count}', number_format($stat_count), $field_config['description']);

                                    $field_type = $field_config['type'] ?? 'checkbox';
                                    ?>

                                    <?php if ($field_type === 'file'): ?>
                                        <input type="file"
                                               id="linkcentral-<?php echo esc_attr($field_key); ?>"
                                               name="<?php echo esc_attr($field_key); ?>"
                                               accept=".csv,.json">
                                        <p class="description">
                                            <?php echo esc_html($description); ?>
                                        </p>
                                    <?php elseif ($field_type === 'radio' && isset($field_config['options'])): ?>
                                        <?php foreach ($field_config['options'] as $option_value => $option_label): ?>
                                            <label style="display: block; margin-bottom: 10px;">
                                                <input type="radio"
                                                       name="linkcentral-<?php echo esc_attr($field_key); ?>"
                                                       value="<?php echo esc_attr($option_value); ?>"
                                                       <?php checked($option_value === 'migration'); // Default to migration ?>>
                                                <?php echo esc_html($option_label); ?>
                                            </label>
                                        <?php endforeach; ?>
                                        <p class="description">
                                            <?php echo esc_html($description); ?>
                                        </p>
                                    <?php else: ?>
                                        <label class="linkcentral-toggle-switch">
                                            <input type="<?php echo esc_attr($field_type); ?>"
                                                   id="linkcentral-<?php echo esc_attr($field_key); ?>"
                                                   name="<?php echo esc_attr($field_key); ?>"
                                                   <?php echo $is_available ? 'checked' : 'disabled'; ?>>
                                            <span class="linkcentral-toggle-slider"></span>
                                        </label>
                                        <span class="linkcentral-toggle-label">
                                            <?php echo esc_html($description); ?>
                                        </span>
                                    <?php endif; ?>

                                    <?php if (isset($field_config['help']) && $field_type !== 'file' && $field_type !== 'radio'): ?>
                                        <p class="description">
                                            <?php echo esc_html($field_config['help']); ?>
                                        </p>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>

                        <tr>
                            <th scope="row">
                                <label for="linkcentral-duplicate-handling"><?php esc_html_e('Duplicate Handling', 'linkcentral'); ?></label>
                            </th>
                            <td>
                                <select id="linkcentral-duplicate-handling" name="duplicate_handling">
                                    <option value="skip"><?php esc_html_e('Skip duplicates (recommended)', 'linkcentral'); ?></option>
                                    <option value="update"><?php esc_html_e('Update existing links', 'linkcentral'); ?></option>
                                </select>
                                <p class="description">
                                    <?php esc_html_e('How to handle links that already exist in LinkCentral (based on slug).', 'linkcentral'); ?>
                                </p>
                            </td>
                        </tr>
                    </table>

                    <div class="linkcentral-import-actions">
                        <button type="submit" class="button button-primary" id="linkcentral-start-import">
                            <?php esc_html_e('Start Import', 'linkcentral'); ?>
                        </button>
                        <a href="<?php echo admin_url('admin.php?page=linkcentral-settings#linkcentral-import-export'); ?>" class="button button-secondary">
                            <?php esc_html_e('Back to Import Options', 'linkcentral'); ?>
                        </a>
                    </div>

                    <?php wp_nonce_field('linkcentral_import_nonce', 'linkcentral_import_nonce'); ?>
                </form>

                <!-- Import Progress -->
                <div id="linkcentral-import-progress" style="display: none;">
                    <h3><?php esc_html_e('Import Progress', 'linkcentral'); ?></h3>
                    
                    <div class="linkcentral-progress-container">
                        <div class="linkcentral-progress-bar">
                            <div class="linkcentral-progress-fill" style="width: 0%;"></div>
                        </div>
                        <div class="linkcentral-progress-text">
                            <span id="linkcentral-progress-current">0</span> / <span id="linkcentral-progress-total">0</span> 
                            (<span id="linkcentral-progress-percentage">0</span>%)
                        </div>
                    </div>
                    
                    <div class="linkcentral-processing-log">
                        <h4><?php esc_html_e('Processing Log', 'linkcentral'); ?></h4>
                        <div id="linkcentral-processing-messages"></div>
                    </div>
                </div>

                <!-- Import Results -->
                <div id="linkcentral-import-results" style="display: none;">
                    <h3><?php esc_html_e('Import Complete', 'linkcentral'); ?></h3>
                    <div class="linkcentral-import-confirmation-summary">
                        <!-- Results will be populated via JavaScript -->
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>
