(function() {
    // Import WordPress i18n function
    const { __ } = wp.i18n;
    
    function initLinkCentralPlugin(tinymce) {
        // Check if linkcentral_tinymce_data is available
        if (typeof linkcentral_tinymce_data !== 'object') {
            console.log('LinkCentral data not found');
            return;
        }

        // Add LinkCentral plugin to the TinyMCE plugin manager
        tinymce.PluginManager.add('linkcentral', function(editor, url) {
            const logoUrl = linkcentral_tinymce_data.plugin_url + 'assets/images/linkcentral-logo.svg';

            // Add a button to the TinyMCE toolbar
            editor.addButton('linkcentral', {
                title: __('Insert LinkCentral Link', 'linkcentral'),
                image: logoUrl,
                onclick: function() {
                    openLinkCentralModal(editor);
                },
                // Make button in the toolbar active when a linkcentral link is selected
                onpostrender: function() {
                    var ctrl = this;
                    editor.on('NodeChange', function(e) {
                        ctrl.active(e.element.nodeName.toLowerCase() === 'a' && (
                            e.element.hasAttribute('data-linkcentral-id-sync') || 
                            e.element.getAttribute('href')?.startsWith('#linkcentral-sync-to-')
                        ));
                    });
                }
            });

            // Custom inline toolbar for LinkCentral links
            var linkCentralToolbar;
            editor.on('preinit', function() {
                if (editor.wp && editor.wp._createToolbar) {
                    linkCentralToolbar = editor.wp._createToolbar([
                        'linkcentral_url_display', // Add URL display
                        'linkcentral_edit',
                        'linkcentral_remove'
                    ], true);
                }
            });

            editor.on('wptoolbar', function(event) {
                var linkNode = editor.dom.getParent(event.element, 'a[data-linkcentral-id-sync], a[href^="#linkcentral-sync-to-"]');

                if (linkNode) {
                    event.element = linkNode;
                    event.toolbar = linkCentralToolbar;
                    updateLinkCentralUrlDisplay(editor, linkNode);
                }
            });

            // Register the URL display control
            editor.addButton('linkcentral_url_display', {
                type: 'container',
                onPostRender: function() {
                    var element = this.getEl();
                    element.innerHTML = '<div class="linkcentral-url-display" style="display: flex; align-items: center;">' + 
                                            '<img src="' + logoUrl + '" style="width: 24px; height: 24px; vertical-align: middle; margin-right: 5px;">' +
                                            '<div>' +
                                                '<div class="linkcentral-url-display-text"></div>' +
                                                '<div class="linkcentral-url-display-id"></div>' +
                                            '</div>' +
                                        '</div>';
                }
            });

            editor.addButton('linkcentral_edit', {
                title: __('Edit LinkCentral Link', 'linkcentral'),
                icon: 'dashicon dashicons-edit',
                onclick: function() {
                    openLinkCentralModal(editor);
                }
            });

            editor.addButton('linkcentral_remove', {
                title: __('Remove LinkCentral Link', 'linkcentral'),
                icon: 'dashicon dashicons-editor-unlink',
                onclick: function() {
                    editor.execCommand('unlink');
                }
            });

            // Function to open the LinkCentral modal dialog
            function openLinkCentralModal(editor) {
                var selectedNode = editor.selection.getNode();
                var linkNode = editor.dom.getParent(selectedNode, 'a[data-linkcentral-id-sync], a[href^="#linkcentral-sync-to-"]');
                
                if (linkNode) {
                    editor.selection.select(linkNode);
                }

                var initialData = {
                    newTab: false,
                    insertAsShortcode: false,
                    parameters: ''
                };

                if (linkNode) {
                    initialData.newTab = linkNode.getAttribute('target') === '_blank';
                    
                    // First check for parameters in the URL
                    const href = linkNode.getAttribute('href');
                    if (href && href.includes('?')) {
                        const urlParts = href.split('?');
                        if (urlParts.length > 1) {
                            initialData.parameters = urlParts[1];
                        }
                    }
                    
                    // If no parameters in URL, check for data attribute (backward compatibility)
                    if (!initialData.parameters && linkNode.getAttribute('data-linkcentral-parameters')) {
                        initialData.parameters = linkNode.getAttribute('data-linkcentral-parameters');
                    }
                    
                    initialData.linkId = linkNode.getAttribute('data-linkcentral-id-sync');
                    if (!initialData.linkId) {
                        const matches = href.match(/#linkcentral-sync-to-(\d+)/);
                        if (matches) {
                            initialData.linkId = matches[1];
                        }
                    }
                }

                // Use global setting for default link insertion type
                // Or set to 'synchronized' if focused link is a synchronized link
                let linkInsertionType = linkNode ? 'synchronized' : (linkcentral_tinymce_data.default_link_insertion_type || 'synchronized');

                editor.windowManager.open({
                    title: linkNode ? __('Edit LinkCentral Link', 'linkcentral') : __('Insert LinkCentral Link', 'linkcentral'),
                    body: [
                        {
                            type: 'container',
                            name: 'searchContainer',
                            html: '<div style="position: relative;">' +
                                  '<input type="text" id="linkcentral-search" placeholder="' + __('Search for a link', 'linkcentral') + '" style="width: 100%; box-sizing: border-box; padding: 8px; border: 1px solid #ccc; background-color: #fff; border-radius: 4px; margin-bottom: 2px;">' +
                                  '<div id="linkcentral-search-results" style="display:none; position: absolute; z-index: 1000; width: 100%; max-height: 200px; overflow-y: auto; border: 1px solid #ccc; background-color: white; margin-top: 5px; border-radius: 4px; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);"></div>' +
                                  '</div>'
                        },
                        {
                            type: 'checkbox',
                            name: 'newTab',
                            label: __('New tab:', 'linkcentral'),
                            checked: initialData.newTab
                        },
                        {
                            type: linkcentral_tinymce_data.can_use_premium_code__premium_only ? 'textbox' : 'container',
                            name: 'parameters',
                            label: __('Parameters:', 'linkcentral'),
                            html: linkcentral_tinymce_data.can_use_premium_code__premium_only ? '' : '<a href="admin.php?page=linkcentral-settings#linkcentral-premium" target="_blank" class="linkcentral-premium-tag">' + __('Premium', 'linkcentral') + '</a>',
                            value: initialData.parameters,
                            tooltip: __('Add parameters to the link (e.g., param1=value1&param2=value2)', 'linkcentral')
                        },
                        {
                            type: 'container',
                            name: 'linkInsertionType',
                            html: `
                                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 5px;">
                                    <label>${__('Link Insertion Type:', 'linkcentral')}</label>
                                    <a href="https://designforwp.com/docs/linkcentral/creating-and-using-links/inserting-links-into-your-content/" target="_blank" style="text-decoration: none; color: #12668A;font-weight: bold;">${__('What is this?', 'linkcentral')}</a>
                                </div>
                                <div id="link-insertion-type-container" style="display: flex; gap: 10px; margin-bottom: 10px;">
                                    ${['synchronized', 'direct', 'shortcode'].map(type => `
                                        <div 
                                            data-type="${type}"
                                            style="
                                                cursor: ${type === 'shortcode' && editor.name === 'core/button' ? 'not-allowed' : 'pointer'};
                                                padding: 10px;
                                                border: ${linkInsertionType === type ? '2px solid #12668A' : '2px solid #ccc'};
                                                border-radius: 4px;
                                                text-align: center;
                                                background-color: ${linkInsertionType === type ? '#e0f7fa' : '#f9f9f9'};
                                                flex: 1;
                                                display: flex;
                                                flex-direction: column;
                                                align-items: center;
                                                justify-content: center;
                                                opacity: ${type === 'shortcode' && editor.name === 'core/button' ? 0.5 : 1};
                                            "
                                        >
                                            <i class="mce-ico mce-i-${type === 'synchronized' ? 'reload' : type === 'direct' ? 'link' : 'code'}" style="font-size: 20px; margin-bottom: 5px;"></i>
                                            <span>${__(type.charAt(0).toUpperCase() + type.slice(1), 'linkcentral')}</span>
                                        </div>
                                    `).join('')}
                                </div>
                            `
                        }
                    ],
                    onsubmit: function(e) {
                        insertLink(editor, e.data, linkInsertionType);
                    },
                    width: 400,
                    height: 250
                });

                // Add event listener for tile clicks
                setTimeout(() => {
                    const container = document.getElementById('link-insertion-type-container');
                    if (container) {
                        container.addEventListener('click', function(event) {
                            const tile = event.target.closest('div[data-type]');
                            if (tile && tile.style.cursor !== 'not-allowed') {
                                linkInsertionType = tile.getAttribute('data-type');
                                Array.from(container.children).forEach(child => {
                                    child.style.border = '2px solid #ccc';
                                    child.style.backgroundColor = '#f9f9f9';
                                });
                                tile.style.border = '2px solid #12668A';
                                tile.style.backgroundColor = '#e0f7fa';
                            }
                        });
                    }
                }, 100);

                // Set focus to the search input field
                // setTimeout(() => {
                //     const firstInput = document.getElementById('linkcentral-search');
                //     if (firstInput) {
                //         firstInput.focus();
                //     }
                // }, 0);

                // Add event listener to the search input field
                setTimeout(function() {
                    var searchInput = document.getElementById('linkcentral-search');
                    if (searchInput) {
                        if (initialData.linkId) {
                            searchInput.disabled = true;
                            // Fetch and display the current link data
                            fetchLinkData(initialData.linkId, function(linkData) {
                                searchInput.disabled = false;
                                if (linkData) {
                                    searchInput.value = linkData.title;
                                    updateSelectedLink(linkData);
                                }
                            });
                        }

                        searchInput.addEventListener('input', debounce(function(e) {
                            var searchTerm = e.target.value;
                            if (searchTerm.length >= 2) {
                                searchLinks(searchTerm);
                            } else {
                                document.getElementById('linkcentral-search-results').style.display = 'none';
                            }
                        }, 300));

                        // Add focus event to show recent links
                        searchInput.addEventListener('focus', function() {
                            var searchTerm = this.value;
                            if (searchTerm.length < 2) {
                                fetchRecentLinks();
                            }
                        });

                        // Add blur event to hide dropdown when clicking outside
                        searchInput.addEventListener('blur', function() {
                            setTimeout(function() {
                                var resultsContainer = document.getElementById('linkcentral-search-results');
                                if (resultsContainer) {
                                    resultsContainer.style.display = 'none';
                                }
                            }, 150);
                        });
                    }
                }, 100);
            }

            // Function to search for links
            function searchLinks(term) {
                wp.apiFetch({
                    path: `/wp/v2/linkcentral_link?search=${encodeURIComponent(term)}&status=publish`
                }).then((links) => {
                    updateSearchResults(links, true);
                }).catch((error) => {
                    console.error('Error searching links:', error);
                    updateSearchResults([], true);
                });
            }

            // Function to fetch recent links
            function fetchRecentLinks() {
                wp.apiFetch({
                    path: `/wp/v2/linkcentral_link?status=publish&orderby=modified&order=desc&per_page=5`
                }).then((links) => {
                    updateSearchResults(links, false);
                }).catch((error) => {
                    console.error('Error fetching recent links:', error);
                    updateSearchResults([], false);
                });
            }

            // Function to update the search results in the modal
            function updateSearchResults(links, isSearch) {
                var resultsContainer = document.getElementById('linkcentral-search-results');
                if (!resultsContainer) {
                    return;
                }

                resultsContainer.innerHTML = '';
                
                // Add header
                var headerElement = document.createElement('div');
                headerElement.style.cssText = 'padding: 8px 12px; background-color: #f8f9fa; border-bottom: 1px solid #e1e5e9; font-size: 12px; font-weight: bold; color: #666; border-top-left-radius: 4px; border-top-right-radius: 4px;';
                headerElement.textContent = isSearch ? 'Search Results' : 'Recently Updated Links';
                resultsContainer.appendChild(headerElement);

                if (links.length > 0) {
                    links.forEach(function(link, index) {
                        var linkElement = document.createElement('div');
                        linkElement.innerHTML = '<span style="font-weight: bold;">' + (link.title.rendered || link.title || '') + '</span> (<span style="font-family:monospace,monospace;font-size:0.9em;vertical-align:middle;">' + (link.slug || '') + '</span>)';
                        linkElement.style.cssText = 'cursor: pointer; padding: 8px 12px; background-color: white; border-bottom: ' + (index < links.length - 1 ? '1px solid #f0f0f0' : 'none') + '; transition: background-color 0.2s ease;';
                        
                        linkElement.addEventListener('mouseover', function() {
                            this.style.backgroundColor = '#f8f9fa';
                        });
                        linkElement.addEventListener('mouseout', function() {
                            this.style.backgroundColor = 'white';
                        });
                        linkElement.addEventListener('mousedown', function(e) {
                            e.preventDefault(); // Prevent blur event from firing
                        });
                        linkElement.addEventListener('click', function() {
                            document.getElementById('linkcentral-search').value = link.title.rendered || link.title || '';
                            resultsContainer.style.display = 'none';
                            updateSelectedLink(link);
                        });
                        resultsContainer.appendChild(linkElement);
                    });
                    resultsContainer.style.display = 'block';
                } else {
                    resultsContainer.style.display = 'none';
                }
            }

            // Function to update the selected link in the modal
            function updateSelectedLink(link) {
                editor.windowManager.getWindows()[0].selectedLink = JSON.stringify(link);
            }

            // Function to insert the link into the TinyMCE editor
            function insertLink(editor, data, linkInsertionType) {
                var selectedLink = JSON.parse(editor.windowManager.getWindows()[0].selectedLink);
                var linkText = editor.selection.getContent({format: 'text'}) || selectedLink.title;

                editor.undoManager.transact(function() {
                    if (linkInsertionType === 'shortcode') {
                        var shortcode = '[linkcentral id="' + selectedLink.id + '"';
                        if (data.newTab) {
                            shortcode += ' newtab="true"';
                        }
                        if (data.parameters) {
                            shortcode += ' parameters="' + data.parameters + '"';
                        }
                        shortcode += ']' + linkText + '[/linkcentral]';
                        editor.insertContent(shortcode);
                    } else if (linkInsertionType === 'direct') {
                        if (!selectedLink.url) {
                            console.error('URL not provided by server for link:', selectedLink.id);
                            return;
                        }
                        
                        var url = selectedLink.url;
                        if (data.parameters) {
                            url += url.includes('?') ? '&' : '?';
                            url += data.parameters;
                        }
                        var directLink = '<a href="' + url + '"';
                        if (data.newTab) {
                            directLink += ' target="_blank"';
                        }

                        // Construct rel attributes
                        var relAttributes = [
                            selectedLink.nofollow === 'yes' ? 'nofollow' : (selectedLink.nofollow === 'default' ? (selectedLink.global_nofollow ? 'nofollow' : '') : ''),
                            selectedLink.sponsored === 'yes' ? 'sponsored' : (selectedLink.sponsored === 'default' ? (selectedLink.global_sponsored ? 'sponsored' : '') : '')
                        ].filter(Boolean).join(' ');

                        if (relAttributes) {
                            directLink += ' rel="' + relAttributes + '"';
                        }

                        // Add CSS classes
                        var cssClasses = selectedLink.css_classes_option === 'replace'
                            ? selectedLink.custom_css_classes
                            : `${selectedLink.global_css_classes || ''} ${selectedLink.custom_css_classes || ''}`.trim();
                        if (cssClasses) {
                            directLink += ' class="' + cssClasses + '"';
                        }

                        directLink += '>' + linkText + '</a>';
                        editor.selection.setContent(directLink);
                    } else { // synchronized
                        createLinkCentralLink(editor, selectedLink, data);
                    }
                });
            }

            // Debounce function to limit the rate at which a function can fire
            function debounce(func, wait) {
                var timeout;
                return function() {
                    var context = this, args = arguments;
                    clearTimeout(timeout);
                    timeout = setTimeout(function() {
                        func.apply(context, args);
                    }, wait);
                };
            }

            // Function to update the URL display in the toolbar
            function updateLinkCentralUrlDisplay(editor, linkNode) {
                // Check if the selected link is the same as the previously selected link
                if (editor.lastSelectedLinkNode === linkNode) {
                    return; // Do not reload the data
                }

                // Store the currently selected link node
                editor.lastSelectedLinkNode = linkNode;

                // Wait for the next tick to ensure the toolbar has been rendered
                setTimeout(function() {
                    // Find all inline toolbars in the document
                    var toolbars = document.querySelectorAll('.mce-inline-toolbar-grp');
                    var activeToolbar;
                    
                    // Find the visible toolbar
                    for (var i = 0; i < toolbars.length; i++) {
                        if (toolbars[i].offsetParent !== null) {
                            activeToolbar = toolbars[i];
                            break;
                        }
                    }

                    if (!activeToolbar) {
                        console.log('Active toolbar not found');
                        return;
                    }

                    var urlDisplayControl = activeToolbar.querySelector('.linkcentral-url-display');
                    if (!urlDisplayControl) {
                        console.log('URL display control not found');
                        return;
                    }

                    var urlDisplayElement = urlDisplayControl.querySelector('.linkcentral-url-display-text');
                    var idDisplayElement = urlDisplayControl.querySelector('.linkcentral-url-display-id');
                    
                    if (urlDisplayElement) {
                        urlDisplayElement.innerHTML = 'Loading preview...';
                    }
                    
                    var linkId = linkNode.getAttribute('data-linkcentral-id-sync');
                    if (!linkId) {
                        const href = linkNode.getAttribute('href');
                        const matches = href.match(/#linkcentral-sync-to-(\d+)/);
                        if (matches) {
                            linkId = matches[1];
                        }
                    }
                    
                    // Extract parameters from the URL if present
                    let parameters = '';
                    const href = linkNode.getAttribute('href');
                    if (href && href.includes('?')) {
                        const urlParts = href.split('?');
                        if (urlParts.length > 1) {
                            parameters = urlParts[1];
                        }
                    }
                    
                    // If no parameters in URL, check for data attribute (backward compatibility)
                    if (!parameters && linkNode.getAttribute('data-linkcentral-parameters')) {
                        parameters = linkNode.getAttribute('data-linkcentral-parameters');
                    }
                    
                    if (idDisplayElement) {
                        idDisplayElement.innerHTML = '<span style="font-size: 10px; font-weight: bold;">Synchronized with ID: ' + linkId + '</span>';
                    }
                    
                    if (linkId) {
                        wp.apiFetch({
                            path: `/wp/v2/linkcentral_link/${linkId}`
                        }).then((linkData) => {
                            if (!linkData.url) {
                                console.error('URL not provided by server for link:', linkData.id);
                                urlDisplayElement.innerHTML = '<span style="color:red;">URL not available</span>';
                                return;
                            }
                            
                            var fullUrl = linkData.url;
                            
                            // Add parameters to the displayed URL if they exist
                            if (parameters) {
                                fullUrl += '?' + parameters;
                            }
                            
                            var editUrl = linkcentral_tinymce_data.site_url + '/wp-admin/post.php?post=' + linkData.id + '&action=edit';
                            
                            if (urlDisplayElement) {
                                urlDisplayElement.innerHTML = '<a href="' + editUrl + '" target="_blank" style="text-decoration: none;">' + fullUrl + '</a>';
                            }
                        }).catch((error) => {
                            console.error('Error loading link data:', error);
                            urlDisplayElement.innerHTML = '<span style="color:red;">Error loading link data</span>';
                        });
                    }
                }, 0);
            }

            // Add custom styles to make the linkcentral element appear with a gradient border at the bottom in the editor
            editor.on('init', function() {
                var head = editor.getDoc().head;
                var style = editor.getDoc().createElement('style');
                style.type = 'text/css';
                style.innerHTML = `
                    a[data-linkcentral-id-sync],
                    a[href^="#linkcentral-sync-to-"] {
                        text-decoration: none;
                        cursor: pointer;
                        border-bottom: 2px solid;
                        border-image: linear-gradient(to right, #12668A, #68C8CB) 1;
                        display: inline-block;
                        padding-bottom: 0;
                        line-height: 1;
                    }
                `;
                head.appendChild(style);
            });
        });

        // Initialize the plugin for existing editors
        if (tinymce.editors.length > 0) {
            tinymce.editors.forEach(function(editor) {
                editor.addCommand('mceLinkCentral', function() {
                    openLinkCentralModal(editor);
                });
            });
        }
    }

    // Function to initialize the plugin when TinyMCE is ready
    function initializeWhenReady() {
        if (typeof window.tinymce !== 'undefined' && window.tinymce.PluginManager) {
            initLinkCentralPlugin(window.tinymce);
        } else {
            // If TinyMCE is not yet available, wait for it to load
            window.addEventListener('tinymce-editor-setup', function() {
                initLinkCentralPlugin(window.tinymce);
            });
        }
    }

    // Check if the DOM is already loaded
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initializeWhenReady);
    } else {
        initializeWhenReady();
    }

    // Also add event listener for when TinyMCE might load later
    if (typeof window.tinymce !== 'undefined') {
        window.tinymce.on('AddEditor', function(e) {
            e.editor.addCommand('mceLinkCentral', function() {
                openLinkCentralModal(e.editor);
            });
        });
    }

    // Add this new function to fetch link data
    function fetchLinkData(linkId, callback) {
        wp.apiFetch({
            path: `/wp/v2/linkcentral_link/${linkId}`
        }).then((linkData) => {
            callback(linkData);
        }).catch((error) => {
            console.error('Error fetching link data:', error);
            callback(null);
        });
    }

    function createLinkCentralLink(editor, selectedLink, data) {
        // Start with the base URL
        let href = `#linkcentral-sync-to-${selectedLink.id}`;
        
        // Append parameters to the URL if they exist
        if (data.parameters) {
            href += href.includes('?') ? '&' : '?';
            href += data.parameters;
        }
        
        // Apply the link
        editor.execCommand('mceInsertLink', false, {
            href: href,
            target: data.newTab ? '_blank' : null,
            rel: data.newTab ? 'noopener noreferrer' : null
        });
        
        // Get the current selection's link node
        const selectedNode = editor.selection.getNode();
        const linkNode = editor.dom.getParent(selectedNode, 'a');
        
        // Explicitly remove the data attributes for the new URL format
        if (linkNode) {
            editor.dom.setAttrib(linkNode, 'data-linkcentral-id-sync', null);
            editor.dom.setAttrib(linkNode, 'data-linkcentral-parameters', null);
        }
    }
})();
