class SharedTabNavigator {
    constructor(containerSelector, onTabChangeCallback = null) {
        this.containerSelector = containerSelector;
        this.onTabChangeCallback = onTabChangeCallback;
        this.currentTab = null;
        this.tabs = [];
    }

    init() {
        const container = document.querySelector(this.containerSelector);
        if (!container) {
            console.warn(`Tab container not found: ${this.containerSelector}`);
            return;
        }

        // Find all tabs within the container
        this.tabs = container.querySelectorAll('.linkcentral-insights-tab');
        
        if (this.tabs.length === 0) {
            console.warn(`No tabs found in container: ${this.containerSelector}`);
            return;
        }

        // Set initial active tab
        const activeTab = container.querySelector('.linkcentral-insights-tab-active');
        if (activeTab) {
            this.currentTab = activeTab.dataset.tab;
        } else if (this.tabs.length > 0) {
            // If no active tab, activate the first one
            this.switchTab(this.tabs[0].dataset.tab);
        }

        // Add event listeners
        this.tabs.forEach(tab => {
            tab.addEventListener('click', (e) => {
                const tabType = e.target.dataset.tab;
                this.switchTab(tabType);
            });
        });
    }

    switchTab(tabType) {
        if (!tabType) return;

        const container = document.querySelector(this.containerSelector);
        if (!container) return;

        // Update active tab
        container.querySelectorAll('.linkcentral-insights-tab').forEach(tab => {
            tab.classList.remove('linkcentral-insights-tab-active');
        });
        
        const targetTab = container.querySelector(`[data-tab="${tabType}"]`);
        if (targetTab) {
            targetTab.classList.add('linkcentral-insights-tab-active');
        }
        
        // Update current tab
        this.currentTab = tabType;
        
        // Call callback if provided
        if (this.onTabChangeCallback && typeof this.onTabChangeCallback === 'function') {
            this.onTabChangeCallback(tabType);
        }
    }

    getCurrentTab() {
        return this.currentTab;
    }

    setCurrentTab(tabType) {
        this.switchTab(tabType);
    }
}

export default SharedTabNavigator; 