<?php

/**
 * LinkCentral User Agent Parser Utility
 *
 * This class provides shared functionality for parsing user agent strings
 * to extract browser, device, and operating system information.
 *
 * Supports both:
 * - Full user agent strings (e.g., "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7)...")
 * - Parsed format (e.g., "Chrome|macOS|desktop") from imported data (e.g. ThirstyAffiliates)
 *
 * Uses substring matching (strpos) for compatibility with both formats.
 */
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class LinkCentral_User_Agent_Parser {
    
    /**
     * Extract browser name from user agent string.
     *
     * @param string $user_agent The user agent string to parse.
     * @param string $default_value The default value to return for empty user agents.
     * @return string The browser name.
     */
    public static function get_browser($user_agent, $default_value = 'Other/Unknown') {
        if (empty($user_agent)) {
            return $default_value;
        }
        
        // Check for browsers in order of specificity
        if (strpos($user_agent, 'Edg/') !== false || strpos($user_agent, 'Edge/') !== false) {
            return 'Edge';
        } elseif (strpos($user_agent, 'Chrome') !== false && strpos($user_agent, 'Edg/') === false) {
            return 'Chrome';
        } elseif (strpos($user_agent, 'Firefox') !== false) {
            return 'Firefox';
        } elseif (strpos($user_agent, 'Safari') !== false && strpos($user_agent, 'Chrome') === false) {
            return 'Safari';
        } elseif (strpos($user_agent, 'MSIE') !== false || strpos($user_agent, 'Trident/') !== false) {
            return 'Internet Explorer';
        } else {
            return 'Other/Unknown';
        }
    }

    /**
     * Extract device type from user agent string.
     *
     * @param string $user_agent The user agent string to parse.
     * @param string $default_value The default value to return for empty user agents.
     * @return string The device type (Mobile, Tablet, Desktop).
     */
    public static function get_device($user_agent, $default_value = 'Other/Unknown') {
        if (empty($user_agent)) {
            return $default_value;
        }
        
        $user_agent = strtolower($user_agent);
        
        // Check for tablets first (more specific)
        if (strpos($user_agent, 'tablet') !== false || 
            strpos($user_agent, 'ipad') !== false ||
            strpos($user_agent, 'kindle') !== false ||
            (strpos($user_agent, 'android') !== false && strpos($user_agent, 'mobile') === false)) {
            return 'Tablet';
        }
        
        // Check for mobile devices
        if (strpos($user_agent, 'mobile') !== false || 
            strpos($user_agent, 'android') !== false ||
            strpos($user_agent, 'iphone') !== false ||
            strpos($user_agent, 'ipod') !== false ||
            strpos($user_agent, 'blackberry') !== false ||
            strpos($user_agent, 'windows phone') !== false ||
            strpos($user_agent, 'silk') !== false ||
            strpos($user_agent, 'opera mini') !== false) {
            return 'Mobile';
        }
        
        // Default to desktop
        return 'Desktop';
    }

    /**
     * Extract operating system from user agent string.
     *
     * @param string $user_agent The user agent string to parse.
     * @param string $default_value The default value to return for empty user agents.
     * @return string The operating system name.
     */
    public static function get_operating_system($user_agent, $default_value = 'Other/Unknown') {
        if (empty($user_agent)) {
            return $default_value;
        }
        
        // Check for operating systems in order of specificity
        if (strpos($user_agent, 'Windows NT 10.0') !== false) {
            return 'Windows 10/11';
        } elseif (strpos($user_agent, 'Windows NT 6.3') !== false) {
            return 'Windows 8.1';
        } elseif (strpos($user_agent, 'Windows NT 6.2') !== false) {
            return 'Windows 8';
        } elseif (strpos($user_agent, 'Windows NT 6.1') !== false) {
            return 'Windows 7';
        } elseif (strpos($user_agent, 'Windows NT') !== false) {
            return 'Windows';
        } elseif (strpos($user_agent, 'Mac OS X') !== false || strpos($user_agent, 'macOS') !== false) {
            return 'macOS';
        } elseif (strpos($user_agent, 'iPhone OS') !== false || strpos($user_agent, 'iOS') !== false) {
            return 'iOS';
        } elseif (strpos($user_agent, 'Android') !== false) {
            return 'Android';
        } elseif (strpos($user_agent, 'Linux') !== false) {
            return 'Linux';
        } elseif (strpos($user_agent, 'CrOS') !== false) {
            return 'Chrome OS';
        } else {
            return 'Other/Unknown';
        }
    }

    /**
     * Get device type, icon, and operating system from user agent string.
     * This method returns an array with device information including WordPress dashicons.
     *
     * @param string $user_agent The user agent string to parse.
     * @param string $default_value The default value to return for empty user agents.
     * @return array Array containing 'type', 'icon', and 'os' keys.
     */
    public static function get_device_and_os($user_agent, $default_value = 'Other/Unknown') {
        if (empty($user_agent)) {
            return array(
                'type' => $default_value,
                'icon' => 'dashicons-desktop',
                'os' => $default_value
            );
        }

        $user_agent_lower = strtolower($user_agent);
        $info = array(
            'type' => 'Desktop',
            'icon' => 'dashicons-desktop',
            'os' => 'Other/Unknown'
        );

        // Determine device type and icon
        if (strpos($user_agent_lower, 'mobile') !== false
            || strpos($user_agent_lower, 'android') !== false
            || strpos($user_agent_lower, 'iphone') !== false
            || strpos($user_agent_lower, 'ipod') !== false
            || strpos($user_agent_lower, 'silk') !== false
            || strpos($user_agent_lower, 'blackberry') !== false
            || strpos($user_agent_lower, 'opera mini') !== false) {
            $info['type'] = 'Mobile';
            $info['icon'] = 'dashicons-smartphone';
        } elseif (strpos($user_agent_lower, 'tablet') !== false
            || strpos($user_agent_lower, 'ipad') !== false
            || strpos($user_agent_lower, 'kindle') !== false) {
            $info['type'] = 'Tablet';
            $info['icon'] = 'dashicons-tablet';
        }

        // Determine OS - use the same logic as get_operating_system method
        if (strpos($user_agent, 'Windows NT 10.0') !== false) {
            $info['os'] = 'Windows 10/11';
        } elseif (strpos($user_agent, 'Windows NT 6.3') !== false) {
            $info['os'] = 'Windows 8.1';
        } elseif (strpos($user_agent, 'Windows NT 6.2') !== false) {
            $info['os'] = 'Windows 8';
        } elseif (strpos($user_agent, 'Windows NT 6.1') !== false) {
            $info['os'] = 'Windows 7';
        } elseif (strpos($user_agent, 'Windows NT') !== false) {
            $info['os'] = 'Windows';
        } elseif (strpos($user_agent, 'Mac OS X') !== false || strpos($user_agent, 'macOS') !== false) {
            $info['os'] = 'macOS';
        } elseif (strpos($user_agent, 'iPhone OS') !== false || strpos($user_agent, 'iOS') !== false) {
            $info['os'] = 'iOS';
        } elseif (strpos($user_agent, 'Android') !== false) {
            $info['os'] = 'Android';
        } elseif (strpos($user_agent, 'Linux') !== false) {
            $info['os'] = 'Linux';
        } elseif (strpos($user_agent, 'CrOS') !== false) {
            $info['os'] = 'Chrome OS';
        }

        return $info;
    }

    /**
     * Get comprehensive user agent information including browser, device, and OS.
     *
     * @param string $user_agent The user agent string to parse.
     * @param string $default_value The default value to return for empty user agents.
     * @return array Array containing browser, device, and OS information.
     */
    public static function get_user_agent_info($user_agent, $default_value = 'Other/Unknown') {
        $device_and_os = self::get_device_and_os($user_agent, $default_value);
        
        return array(
            'browser' => self::get_browser($user_agent, $default_value),
            'device' => $device_and_os['type'],
            'device_icon' => $device_and_os['icon'],
            'os' => $device_and_os['os']
        );
    }
} 