<?php

/**
 * LinkCentral Settings Class
 *
 * This class handles the settings functionality,
 * including registering settings, rendering the settings page, and handling
 * settings updates.
 */
if ( !defined( 'ABSPATH' ) ) {
    exit;
}
// Include the import/export functionality
require_once LINKCENTRAL_PLUGIN_DIR . 'includes/import/core.php';
class LinkCentral_Settings {
    /**
     * Import instance
     */
    private $import;

    /**
     * Export instance
     */
    private $export;

    /**
     * Constructor
     */
    public function __construct() {
        // Initialize import instance immediately
        $this->import = new LinkCentral_Import();
    }

    /**
     * Initialize the class and set up WordPress hooks.
     */
    public function init() {
        // Initialize import functionality (AJAX hooks, etc.)
        $this->import->init();
        // Register actions for settings and AJAX
        add_action( 'admin_init', array($this, 'register_settings') );
        // Enqueue scripts for settings page
        add_action( 'admin_enqueue_scripts', array($this, 'enqueue_settings_scripts') );
        // Enqueue color picker for QR code settings
        add_action( 'admin_enqueue_scripts', array($this, 'enqueue_color_picker') );
        // Register the new setting
        register_setting( 'linkcentral_settings', 'linkcentral_enable_ga', 'intval' );
    }

    /**
     * Enqueue color picker for admin pages
     */
    public function enqueue_color_picker() {
        wp_enqueue_style( 'wp-color-picker' );
        wp_enqueue_script( 'wp-color-picker' );
    }

    /**
     * Enqueue scripts and styles specific to the settings page
     *
     * @param string $hook The current admin page
     */
    public function enqueue_settings_scripts( $hook ) {
        // Only enqueue on the settings page
        if ( $hook !== 'linkcentral_page_linkcentral-settings' ) {
            return;
        }
    }

    /**
     * Register all settings for LinkCentral.
     */
    public function register_settings() {
        // Register various settings for LinkCentral
        register_setting( 'linkcentral_settings', 'linkcentral_url_prefix', array(
            'sanitize_callback' => 'sanitize_title',
        ) );
        register_setting( 'linkcentral_settings', 'linkcentral_case_sensitive_redirects', 'boolval' );
        register_setting( 'linkcentral_settings', 'linkcentral_excluded_ips' );
        register_setting( 'linkcentral_settings', 'linkcentral_excluded_roles' );
        register_setting( 'linkcentral_settings', 'linkcentral_disable_reporting', 'boolval' );
        register_setting( 'linkcentral_settings', 'linkcentral_enable_data_expiry', 'boolval' );
        register_setting( 'linkcentral_settings', 'linkcentral_data_expiry_days', 'intval' );
        register_setting( 'linkcentral_settings', 'linkcentral_global_nofollow', 'boolval' );
        register_setting( 'linkcentral_settings', 'linkcentral_global_sponsored', 'boolval' );
        register_setting( 'linkcentral_settings', 'linkcentral_global_redirection_type', 'sanitize_text_field' );
        register_setting( 'linkcentral_settings', 'linkcentral_exclude_bots', 'boolval' );
        register_setting( 'linkcentral_settings', 'linkcentral_track_user_agent', 'boolval' );
        register_setting( 'linkcentral_settings', 'linkcentral_ga_measurement_id', 'sanitize_text_field' );
        register_setting( 'linkcentral_settings', 'linkcentral_ga_api_secret', 'sanitize_text_field' );
        register_setting( 'linkcentral_settings', 'linkcentral_delete_tracking_data_on_link_deletion', 'boolval' );
        register_setting( 'linkcentral_settings', 'linkcentral_track_unique_visitors', 'boolval' );
        register_setting( 'linkcentral_settings', 'linkcentral_default_link_insertion_type', 'sanitize_text_field' );
        register_setting( 'linkcentral_settings', 'linkcentral_enable_qr_codes', 'boolval' );
        register_setting( 'linkcentral_settings', 'linkcentral_qr_background_color', 'sanitize_text_field' );
        register_setting( 'linkcentral_settings', 'linkcentral_qr_background_transparent', 'boolval' );
        register_setting( 'linkcentral_settings', 'linkcentral_qr_front_color', 'sanitize_text_field' );
        register_setting( 'linkcentral_settings', 'linkcentral_enable_dashboard_widget', 'boolval' );
        register_setting( 'linkcentral_settings', 'linkcentral_dashboard_widget_period', 'sanitize_text_field' );
    }

    /**
     * Render the settings page.
     */
    public function render_settings_page() {
        // Check if settings are being saved
        if ( isset( $_POST['submit'] ) && check_admin_referer( 'linkcentral_save_settings', 'linkcentral_settings_nonce' ) ) {
            $this->save_settings();
        }
        // Handle reset clicks data
        if ( isset( $_POST['linkcentral_reset_clicks'] ) && check_admin_referer( 'linkcentral_save_settings', 'linkcentral_settings_nonce' ) ) {
            $this->handle_reset_clicks();
        }
        // Handle reset all data
        if ( isset( $_POST['linkcentral_reset_all'] ) && check_admin_referer( 'linkcentral_save_settings', 'linkcentral_settings_nonce' ) ) {
            $this->handle_reset_all();
        }
        // Get current settings
        $url_prefix = get_option( 'linkcentral_url_prefix', 'go' );
        $case_sensitive_redirects = get_option( 'linkcentral_case_sensitive_redirects', false );
        $excluded_roles = get_option( 'linkcentral_excluded_roles', array() );
        $disable_reporting = get_option( 'linkcentral_disable_reporting', false );
        $enable_data_expiry = get_option( 'linkcentral_enable_data_expiry', false );
        $data_expiry_days = get_option( 'linkcentral_data_expiry_days', 90 );
        $global_nofollow = get_option( 'linkcentral_global_nofollow', false );
        $global_sponsored = get_option( 'linkcentral_global_sponsored', false );
        $global_redirection_type = get_option( 'linkcentral_global_redirection_type', '307' );
        $exclude_bots = get_option( 'linkcentral_exclude_bots', false );
        $track_user_agent = get_option( 'linkcentral_track_user_agent', true );
        $global_parameter_forwarding = get_option( 'linkcentral_global_parameter_forwarding', false );
        $delete_tracking_data_on_link_deletion = get_option( 'linkcentral_delete_tracking_data_on_link_deletion', true );
        $custom_css_classes = get_option( 'linkcentral_custom_css_classes', '' );
        $default_disable_prefix = get_option( 'linkcentral_default_disable_prefix', false );
        // Get the active tab from the form submission or default to 'linkcentral-general'
        $active_tab = ( isset( $_POST['active_tab'] ) ? sanitize_text_field( wp_unslash( $_POST['active_tab'] ) ) : 'linkcentral-general' );
        // Include the settings page template
        include LINKCENTRAL_PLUGIN_DIR . 'views/settings-page.php';
    }

    /**
     * Get the import instance.
     *
     * @return LinkCentral_Import The import instance.
     */
    public function get_import_instance() {
        return $this->import;
    }

    /**
     * Save the settings from the settings page form submission.
     */
    private function save_settings() {
        // Add nonce verification at the beginning of the method
        if ( !isset( $_POST['linkcentral_settings_nonce'] ) || !wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['linkcentral_settings_nonce'] ) ), 'linkcentral_save_settings' ) ) {
            wp_die( 'Security check failed. Please try again.' );
        }
        // Handle URL prefix changes
        $old_prefix = get_option( 'linkcentral_url_prefix', 'go' );
        $new_prefix = ( isset( $_POST['linkcentral_url_prefix'] ) ? sanitize_title( wp_unslash( $_POST['linkcentral_url_prefix'] ) ) : $old_prefix );
        if ( $old_prefix !== $new_prefix ) {
            update_option( 'linkcentral_url_prefix', $new_prefix );
        }
        // Update default disable prefix setting
        update_option( 'linkcentral_default_disable_prefix', isset( $_POST['linkcentral_default_disable_prefix'] ) );
        // Update case-sensitive redirects setting
        update_option( 'linkcentral_case_sensitive_redirects', isset( $_POST['linkcentral_case_sensitive_redirects'] ) );
        // Update various boolean settings
        update_option( 'linkcentral_enable_data_expiry', isset( $_POST['linkcentral_enable_data_expiry'] ) );
        // Only update the expiry days if data expiry is enabled and the value is set
        if ( isset( $_POST['linkcentral_enable_data_expiry'] ) && isset( $_POST['linkcentral_data_expiry_days'] ) ) {
            update_option( 'linkcentral_data_expiry_days', intval( $_POST['linkcentral_data_expiry_days'] ) );
        }
        update_option( 'linkcentral_global_nofollow', isset( $_POST['linkcentral_global_nofollow'] ) );
        update_option( 'linkcentral_global_sponsored', isset( $_POST['linkcentral_global_sponsored'] ) );
        // Handle the redirection type
        $redirection_type = ( isset( $_POST['linkcentral_global_redirection_type'] ) ? sanitize_text_field( wp_unslash( $_POST['linkcentral_global_redirection_type'] ) ) : '307' );
        $allowed_types = array('307', '302', '301');
        $redirection_type = ( in_array( $redirection_type, $allowed_types ) ? $redirection_type : '307' );
        update_option( 'linkcentral_global_redirection_type', $redirection_type );
        update_option( 'linkcentral_disable_reporting', isset( $_POST['linkcentral_disable_reporting'] ) );
        // Update tracking settings only if Disable Reporting is not checked
        if ( !isset( $_POST['linkcentral_disable_reporting'] ) ) {
            update_option( 'linkcentral_track_user_agent', isset( $_POST['linkcentral_track_user_agent'] ) );
            update_option( 'linkcentral_track_unique_visitors', isset( $_POST['linkcentral_track_unique_visitors'] ) );
            // Update excluded IPs
            $excluded_ips = ( isset( $_POST['linkcentral_excluded_ips'] ) ? sanitize_textarea_field( wp_unslash( $_POST['linkcentral_excluded_ips'] ) ) : '' );
            update_option( 'linkcentral_excluded_ips', $excluded_ips );
            // Update excluded roles
            $excluded_roles = ( isset( $_POST['linkcentral_excluded_roles'] ) ? array_map( 'sanitize_text_field', wp_unslash( (array) $_POST['linkcentral_excluded_roles'] ) ) : array() );
            update_option( 'linkcentral_excluded_roles', $excluded_roles );
            update_option( 'linkcentral_exclude_bots', isset( $_POST['linkcentral_exclude_bots'] ) );
            // GA4 settings validation
            $enable_ga = isset( $_POST['linkcentral_enable_ga'] );
            update_option( 'linkcentral_enable_ga', $enable_ga );
            $measurement_id = ( isset( $_POST['linkcentral_ga_measurement_id'] ) ? sanitize_text_field( wp_unslash( $_POST['linkcentral_ga_measurement_id'] ) ) : '' );
            $api_secret = ( isset( $_POST['linkcentral_ga_api_secret'] ) ? sanitize_text_field( wp_unslash( $_POST['linkcentral_ga_api_secret'] ) ) : '' );
            if ( $enable_ga ) {
                if ( empty( $measurement_id ) || empty( $api_secret ) ) {
                    add_settings_error(
                        'linkcentral_messages',
                        'linkcentral_ga4_error',
                        __( 'Error: Google Analytics 4 Measurement ID and API Secret are required when Google Analytics integration is enabled.', 'linkcentral' ),
                        'error'
                    );
                } else {
                    update_option( 'linkcentral_ga_measurement_id', $measurement_id );
                    update_option( 'linkcentral_ga_api_secret', $api_secret );
                }
            }
        }
        update_option( 'linkcentral_delete_tracking_data_on_link_deletion', isset( $_POST['linkcentral_delete_tracking_data_on_link_deletion'] ) );
        // Update QR code generation setting
        update_option( 'linkcentral_enable_qr_codes', isset( $_POST['linkcentral_enable_qr_codes'] ) );
        // Update QR code color settings
        if ( isset( $_POST['linkcentral_qr_background_color'] ) ) {
            $qr_bg_color = sanitize_text_field( wp_unslash( $_POST['linkcentral_qr_background_color'] ) );
            update_option( 'linkcentral_qr_background_color', $qr_bg_color );
        }
        update_option( 'linkcentral_qr_background_transparent', isset( $_POST['linkcentral_qr_background_transparent'] ) );
        if ( isset( $_POST['linkcentral_qr_front_color'] ) ) {
            $qr_front_color = sanitize_text_field( wp_unslash( $_POST['linkcentral_qr_front_color'] ) );
            update_option( 'linkcentral_qr_front_color', $qr_front_color );
        }
        // Update dashboard widget settings
        $dashboard_widget_enabled = isset( $_POST['linkcentral_enable_dashboard_widget'] );
        update_option( 'linkcentral_enable_dashboard_widget', $dashboard_widget_enabled );
        // Update dashboard widget period setting if provided
        if ( isset( $_POST['linkcentral_dashboard_widget_period'] ) ) {
            $period = sanitize_text_field( wp_unslash( $_POST['linkcentral_dashboard_widget_period'] ) );
            $allowed_periods = array('7', '14', '30');
            $period = ( in_array( $period, $allowed_periods ) ? $period : '7' );
            update_option( 'linkcentral_dashboard_widget_period', $period );
        }
        // Schedule or unschedule the cleanup event based on the setting
        if ( isset( $_POST['linkcentral_enable_data_expiry'] ) ) {
            LinkCentral_Cleanup::schedule_cleanup();
        } else {
            LinkCentral_Cleanup::deactivate_cleanup();
        }
        // Save the default link insertion type
        $default_link_insertion_type = ( isset( $_POST['linkcentral_default_link_insertion_type'] ) ? sanitize_text_field( wp_unslash( $_POST['linkcentral_default_link_insertion_type'] ) ) : 'synchronized' );
        update_option( 'linkcentral_default_link_insertion_type', $default_link_insertion_type );
        // Save role capabilities
        if ( isset( $_POST['linkcentral_capabilities'] ) && is_array( $_POST['linkcentral_capabilities'] ) ) {
            $submitted_capabilities = $this->sanitize_capabilities( wp_unslash( $_POST['linkcentral_capabilities'] ) );
            // Apply the capabilities to roles directly
            $this->apply_role_capabilities( $submitted_capabilities );
        } else {
            // If no capabilities submitted, remove all capabilities from non-admin roles
            $this->apply_role_capabilities( array() );
        }
        // Only add the success message if there are no errors
        if ( !get_settings_errors( 'linkcentral_messages' ) ) {
            add_settings_error(
                'linkcentral_messages',
                'linkcentral_message',
                __( 'Settings saved successfully.', 'linkcentral' ),
                'updated'
            );
        }
    }

    /**
     * Get a list of preset URL prefixes.
     * 
     * @return array Array of preset URL prefixes.
     */
    public function get_preset_prefixes() {
        // Return an array of preset URL prefixes
        return [
            'go',
            'refer',
            'link',
            'goto',
            'click',
            'recommend',
            'redirect',
            'ext'
        ];
    }

    /**
     * Sanitize capabilities settings.
     *
     * @param array $input The raw input from the form.
     * @return array Sanitized array of role capabilities.
     */
    public function sanitize_capabilities( $input ) {
        $sanitized_input = array();
        if ( !is_array( $input ) ) {
            return $sanitized_input;
        }
        foreach ( $input as $role => $capabilities ) {
            $sanitized_role = sanitize_text_field( $role );
            $sanitized_input[$sanitized_role] = array();
            if ( is_array( $capabilities ) ) {
                foreach ( $capabilities as $cap ) {
                    $sanitized_cap = sanitize_text_field( $cap );
                    // Only add valid capabilities
                    if ( in_array( $sanitized_cap, array(
                        'linkcentral_manage_settings',
                        'linkcentral_view_links',
                        'linkcentral_create_edit_links',
                        'linkcentral_view_stats'
                    ) ) ) {
                        $sanitized_input[$sanitized_role][] = $sanitized_cap;
                    }
                }
            }
        }
        return $sanitized_input;
    }

    /**
     * Apply role capabilities to WordPress roles.
     *
     * @param array $role_capabilities Array of role => capabilities assignments.
     */
    private function apply_role_capabilities( $role_capabilities ) {
        global $wp_roles;
        if ( !isset( $wp_roles ) ) {
            $wp_roles = new WP_Roles();
        }
        // List of all LinkCentral capabilities
        $all_capabilities = array(
            'linkcentral_manage_settings',
            'linkcentral_view_links',
            'linkcentral_create_edit_links',
            'linkcentral_view_stats'
        );
        // For each role in WordPress
        foreach ( $wp_roles->role_names as $role_name => $role_display_name ) {
            // Skip administrator as we don't want to modify their capabilities
            if ( $role_name === 'administrator' ) {
                continue;
            }
            $role = get_role( $role_name );
            if ( !$role ) {
                continue;
            }
            // If role exists in our settings
            if ( isset( $role_capabilities[$role_name] ) && is_array( $role_capabilities[$role_name] ) ) {
                // Add the capabilities that are in our settings
                foreach ( $role_capabilities[$role_name] as $cap ) {
                    if ( in_array( $cap, $all_capabilities ) ) {
                        $role->add_cap( $cap );
                    }
                }
                // Remove capabilities that are not in our settings, except for editor's view links capability
                foreach ( $all_capabilities as $cap ) {
                    if ( !in_array( $cap, $role_capabilities[$role_name] ) && !($role_name === 'editor' && $cap === 'linkcentral_view_links') ) {
                        $role->remove_cap( $cap );
                    }
                }
            } else {
                // If role doesn't exist in our settings, remove all LinkCentral capabilities
                // except for editor's view links capability
                foreach ( $all_capabilities as $cap ) {
                    if ( !($role_name === 'editor' && $cap === 'linkcentral_view_links') ) {
                        $role->remove_cap( $cap );
                    }
                }
            }
            // Always ensure editor has view links capability
            if ( $role_name === 'editor' ) {
                $role->add_cap( 'linkcentral_view_links' );
            }
        }
    }

    /**
     * Sanitize UTM parameter values while preserving URL-valid characters
     * 
     * @param string $value The UTM parameter value to sanitize
     * @return string Sanitized UTM parameter value
     */
    private function sanitize_utm_value( $value ) {
        // Remove any HTML tags and scripts for security
        $value = wp_strip_all_tags( $value );
        // Trim whitespace
        $value = trim( $value );
        // Allow realistic characters for UTM parameters:
        // - Alphanumeric: a-z, A-Z, 0-9
        // - Common separators: hyphens, underscores, periods
        // - Spaces (will be converted to + in URLs)
        // - Plus signs (space replacement in query strings)
        // - Ampersands (sometimes used in campaign names)
        // - Percent signs (for discount campaigns like "50%off")
        $value = preg_replace( '/[^a-zA-Z0-9\\s\\-_.+&%]/', '', $value );
        // Convert multiple spaces to single spaces
        $value = preg_replace( '/\\s+/', ' ', $value );
        return $value;
    }

    /**
     * Handle reset clicks data form submission
     */
    private function handle_reset_clicks() {
        // Check user capabilities
        if ( !current_user_can( 'manage_options' ) ) {
            add_settings_error(
                'linkcentral_messages',
                'linkcentral_reset_error',
                __( 'You do not have permission to perform this action.', 'linkcentral' ),
                'error'
            );
            return;
        }
        global $wpdb;
        $table_name = $wpdb->prefix . 'linkcentral_stats';
        // Delete all click data
        $result = $wpdb->query( "DELETE FROM {$table_name}" );
        if ( $result !== false ) {
            add_settings_error(
                'linkcentral_messages',
                'linkcentral_reset_success',
                /* translators: %d: Number of records deleted */
                sprintf( __( 'All clicking data has been successfully reset. %d records deleted.', 'linkcentral' ), $result ),
                'updated'
            );
        } else {
            add_settings_error(
                'linkcentral_messages',
                'linkcentral_reset_error',
                __( 'Failed to reset clicking data. Please try again.', 'linkcentral' ),
                'error'
            );
        }
    }

    /**
     * Handle reset all data form submission
     */
    private function handle_reset_all() {
        // Check user capabilities
        if ( !current_user_can( 'manage_options' ) ) {
            add_settings_error(
                'linkcentral_messages',
                'linkcentral_reset_error',
                __( 'You do not have permission to perform this action.', 'linkcentral' ),
                'error'
            );
            return;
        }
        global $wpdb;
        // Start a transaction to ensure data consistency
        $wpdb->query( 'START TRANSACTION' );
        $success = true;
        $errors = array();
        $stats = array();
        try {
            // 1. Delete all click data
            $stats_table = $wpdb->prefix . 'linkcentral_stats';
            $clicks_deleted = $wpdb->query( "DELETE FROM {$stats_table}" );
            if ( $clicks_deleted === false ) {
                $success = false;
                $errors[] = __( 'Failed to delete click data', 'linkcentral' );
            } else {
                $stats['clicks'] = $clicks_deleted;
            }
            // 2. Delete all keywords
            $keywords_table = $wpdb->prefix . 'linkcentral_keywords';
            $keywords_deleted = $wpdb->query( "DELETE FROM {$keywords_table}" );
            if ( $keywords_deleted === false ) {
                $success = false;
                $errors[] = __( 'Failed to delete keywords', 'linkcentral' );
            } else {
                $stats['keywords'] = $keywords_deleted;
            }
            // 3. Delete all link posts
            $links_deleted = $wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->posts} WHERE post_type = %s", 'linkcentral_link' ) );
            if ( $links_deleted === false ) {
                $success = false;
                $errors[] = __( 'Failed to delete links', 'linkcentral' );
            } else {
                $stats['links'] = $links_deleted;
            }
            // 4. Delete all link postmeta
            $meta_deleted = $wpdb->query( $wpdb->prepare( "DELETE pm FROM {$wpdb->postmeta} pm\n                 INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID\n                 WHERE p.post_type = %s", 'linkcentral_link' ) );
            if ( $meta_deleted === false ) {
                $success = false;
                $errors[] = __( 'Failed to delete link metadata', 'linkcentral' );
            } else {
                $stats['meta'] = $meta_deleted;
            }
            // 5. Delete all taxonomy terms for linkcentral_category
            $term_deleted = $wpdb->query( $wpdb->prepare( "DELETE tt, t FROM {$wpdb->term_taxonomy} tt\n                 LEFT JOIN {$wpdb->terms} t ON tt.term_id = t.term_id\n                 WHERE tt.taxonomy = %s", 'linkcentral_category' ) );
            if ( $term_deleted === false ) {
                $success = false;
                $errors[] = __( 'Failed to delete categories', 'linkcentral' );
            } else {
                $stats['categories'] = $term_deleted;
            }
            // 6. Delete all term relationships for links
            $relationships_deleted = $wpdb->query( $wpdb->prepare( "DELETE tr FROM {$wpdb->term_relationships} tr\n                 INNER JOIN {$wpdb->posts} p ON tr.object_id = p.ID\n                 WHERE p.post_type = %s", 'linkcentral_link' ) );
            if ( $relationships_deleted === false ) {
                $success = false;
                $errors[] = __( 'Failed to delete category relationships', 'linkcentral' );
            } else {
                $stats['relationships'] = $relationships_deleted;
            }
            // 7. Clear all broken links checker data
            if ( class_exists( 'LinkCentral_Broken_Links_Checker' ) ) {
                LinkCentral_Broken_Links_Checker::clear_all_broken_links_data();
            }
            if ( $success ) {
                $wpdb->query( 'COMMIT' );
                add_settings_error(
                    'linkcentral_messages',
                    'linkcentral_reset_success',
                    __( 'All LinkCentral data has been successfully reset.', 'linkcentral' ),
                    'updated'
                );
            } else {
                $wpdb->query( 'ROLLBACK' );
                add_settings_error(
                    'linkcentral_messages',
                    'linkcentral_reset_error',
                    __( 'Failed to reset all data. Some operations failed.', 'linkcentral' ) . ' ' . implode( ', ', $errors ),
                    'error'
                );
            }
        } catch ( Exception $e ) {
            $wpdb->query( 'ROLLBACK' );
            add_settings_error(
                'linkcentral_messages',
                'linkcentral_reset_error',
                __( 'An error occurred during the reset process.', 'linkcentral' ),
                'error'
            );
        }
    }

}
