<?php
/**
 * LinkCentral Migration Notices Class
 *
 * Handles admin notices for plugin migrations (ThirstyAffiliates, etc.)
 */
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// Include the import handlers for plugin detection
require_once plugin_dir_path( __FILE__ ) . '../import/handlers/thirstyaffiliates.php';
require_once plugin_dir_path( __FILE__ ) . '../import/handlers/prettylinks.php';

class LinkCentral_Migration_Notices {
    /**
     * Import admin instance
     */
    private $import_admin;

    /**
     * Constructor
     */
    public function __construct($import_admin) {
        $this->import_admin = $import_admin;

        // Initialize hooks
        add_action('admin_notices', array($this, 'show_thirstyaffiliates_migration_notice'));
        add_action('admin_notices', array($this, 'show_prettylinks_migration_notice'));
        add_action('wp_ajax_linkcentral_dismiss_thirstyaffiliates_notice', array($this, 'dismiss_thirstyaffiliates_notice'));
        add_action('wp_ajax_linkcentral_dismiss_prettylinks_notice', array($this, 'dismiss_prettylinks_notice'));
    }

    /**
     * Show admin notice for ThirstyAffiliates migration
     */
    public function show_thirstyaffiliates_migration_notice() {
        // Only show to users who can manage LinkCentral settings
        if (!current_user_can('linkcentral_manage_settings')) {
            return;
        }

        // Don't show on user profile pages or the import page itself
        $screen = get_current_screen();
        if ($screen && (strpos($screen->id, 'profile') !== false || strpos($screen->id, 'linkcentral-thirstyaffiliates-import') !== false)) {
            return;
        }

        // Check if notice has been dismissed site-wide (affects all users)
        $dismissed = get_option('linkcentral_dismissed_thirstyaffiliates_notice', false);
        if ($dismissed) {
            return;
        }

        // Check if ThirstyAffiliates is available (active and has data)
        if (!LinkCentral_ThirstyAffiliates_Utils::is_available()) {
            return;
        }

        ?>
        <div id="linkcentral-thirstyaffiliates-notice" class="notice notice-info">
            <p>
                <strong><?php esc_html_e('LinkCentral:', 'linkcentral'); ?></strong>
                <?php esc_html_e('We detected ThirstyAffiliates is active on your site! You can easily migrate your affiliate links, categories, and click statistics to LinkCentral.', 'linkcentral'); ?>
            </p>
            <p>
                <a href="<?php echo admin_url('admin.php?page=linkcentral-thirstyaffiliates-import'); ?>" class="button button-primary">
                    <?php esc_html_e('Migrate from ThirstyAffiliates', 'linkcentral'); ?>
                </a>
                <a href="#" class="linkcentral-dismiss-notice" data-notice="thirstyaffiliates" style="margin-left: 10px;">
                    <?php esc_html_e('Don\'t show again', 'linkcentral'); ?>
                </a>
            </p>
        </div>

        <?php
    }

    /**
     * Show admin notice for Pretty Links migration
     */
    public function show_prettylinks_migration_notice() {
        // Only show to users who can manage LinkCentral settings
        if (!current_user_can('linkcentral_manage_settings')) {
            return;
        }

        // Don't show on user profile pages or the import page itself
        $screen = get_current_screen();
        if ($screen && (strpos($screen->id, 'profile') !== false || strpos($screen->id, 'linkcentral-prettylinks-import') !== false)) {
            return;
        }

        // Check if notice has been dismissed site-wide (affects all users)
        $dismissed = get_option('linkcentral_dismissed_prettylinks_notice', false);
        if ($dismissed) {
            return;
        }

        // Check if Pretty Links is available (active and has data)
        if (!LinkCentral_PrettyLinks_Utils::is_available()) {
            return;
        }

        ?>
        <div id="linkcentral-prettylinks-notice" class="notice notice-info">
            <p>
                <strong><?php esc_html_e('LinkCentral:', 'linkcentral'); ?></strong>
                <?php esc_html_e('We detected Pretty Links is active on your site! You can easily migrate your affiliate links, categories, and click statistics to LinkCentral.', 'linkcentral'); ?>
            </p>
            <p>
                <a href="<?php echo admin_url('admin.php?page=linkcentral-prettylinks-import'); ?>" class="button button-primary">
                    <?php esc_html_e('Migrate from Pretty Links', 'linkcentral'); ?>
                </a>
                <a href="#" class="linkcentral-dismiss-notice" data-notice="prettylinks" style="margin-left: 10px;">
                    <?php esc_html_e('Don\'t show again', 'linkcentral'); ?>
                </a>
            </p>
        </div>

        <?php
    }

    /**
     * Handle AJAX request to dismiss ThirstyAffiliates notice
     */
    public function dismiss_thirstyaffiliates_notice() {
        // Only allow users who can manage LinkCentral settings
        if (!current_user_can('linkcentral_manage_settings')) {
            wp_send_json_error(__('Insufficient permissions', 'linkcentral'));
        }

        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'] ?? '', 'linkcentral_dismiss_notice')) {
            wp_send_json_error(__('Security check failed', 'linkcentral'));
        }

        // Dismiss site-wide (affects all users)
        update_option('linkcentral_dismissed_thirstyaffiliates_notice', '1');

        wp_send_json_success();
    }

    /**
     * Handle AJAX request to dismiss Pretty Links notice
     */
    public function dismiss_prettylinks_notice() {
        // Only allow users who can manage LinkCentral settings
        if (!current_user_can('linkcentral_manage_settings')) {
            wp_send_json_error(__('Insufficient permissions', 'linkcentral'));
        }

        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'] ?? '', 'linkcentral_dismiss_notice')) {
            wp_send_json_error(__('Security check failed', 'linkcentral'));
        }

        // Dismiss site-wide (affects all users)
        update_option('linkcentral_dismissed_prettylinks_notice', '1');

        wp_send_json_success();
    }
}
