<?php
/**
 * LinkCentral Import Admin Class
 *
 * This class handles the admin-side functionality for import pages.
 */
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

require_once plugin_dir_path( __FILE__ ) . '../import/handlers/thirstyaffiliates.php';
require_once plugin_dir_path( __FILE__ ) . '../import/handlers/prettylinks.php';

class LinkCentral_Import_Admin {
    /**
     * The import instance.
     */
    private $import;

    /**
     * Constructor.
     *
     * @param LinkCentral_Import $import The import instance.
     */
    public function __construct($import) {
        $this->import = $import;
    }

    /**
     * Render the ThirstyAffiliates import page.
     */
    public function render_thirstyaffiliates_import_page() {
        // Get stats (includes availability check and caching)
        $stats = LinkCentral_ThirstyAffiliates_Utils::get_stats();
        $is_plugin_installed = $stats['links'] > 0;

        // Get prefix information
        $ta_prefix = LinkCentral_ThirstyAffiliates_Utils::get_thirstyaffiliates_prefix();
        $lc_prefix = get_option('linkcentral_url_prefix', 'go');
        $prefixes_different = ($ta_prefix !== $lc_prefix && !empty($ta_prefix));

        // Set up template variables
        $import_config = array(
            'type' => 'thirstyaffiliates',
            'plugin_name' => 'ThirstyAffiliates',
            'stats' => array(
                'links' => array('label' => __('Links', 'linkcentral')),
                'categories' => array('label' => __('Categories', 'linkcentral')),
                'stats' => array('label' => __('Click Records', 'linkcentral')),
            ),
            'form' => array(),
        );

        // Conditionally add the prefix update option
        if ($prefixes_different) {
            $import_config['form']['update_url_prefix'] = array(
                'type' => 'checkbox',
                'label' => __('Update URL Prefix', 'linkcentral'),
                /* translators: 1: Current LinkCentral URL prefix, 2: ThirstyAffiliates URL prefix */
                'description' => sprintf(__('Update LinkCentral URL prefix from "%1$s" to "%2$s" to match ThirstyAffiliates', 'linkcentral'), $lc_prefix, $ta_prefix),
                'help' => __('This will change your LinkCentral URL prefix to match ThirstyAffiliates.', 'linkcentral'),
            );
        }

        // Always add the other options
        $import_config['form']['import_categories'] = array(
            'type' => 'checkbox',
            'label' => __('Import Categories', 'linkcentral'),
            /* translators: %s: Number of categories */
            'description' => sprintf(__('Import %s categories from ThirstyAffiliates', 'linkcentral'), '{count}'),
            'help' => __('Categories will be created in LinkCentral if they don\'t exist and assigned to the links.', 'linkcentral'),
        );

        $import_config['form']['import_stats'] = array(
            'type' => 'checkbox',
            'label' => __('Import Click Statistics', 'linkcentral'),
            /* translators: %s: Number of click records */
            'description' => sprintf(__('Import %s click records from ThirstyAffiliates', 'linkcentral'), '{count}'),
            'help' => __('This will import historical click data including IP addresses, user agents, and referrers.', 'linkcentral'),
        );

        // Include the migrate import page template
        include LINKCENTRAL_PLUGIN_DIR . 'views/import-migrate.php';
    }

    /**
     * Register all available import pages in the admin menu.
     */
    public function register_import_pages() {
        // Define available import types
        $import_types = array(
            'csv' => array(
                'title' => __('Import from CSV', 'linkcentral'),
                'method' => 'render_csv_import_page'
            ),
            'json' => array(
                'title' => __('Import from JSON', 'linkcentral'),
                'method' => 'render_json_import_page'
            ),
            'thirstyaffiliates' => array(
                'title' => __('Import from ThirstyAffiliates', 'linkcentral'),
                'method' => 'render_thirstyaffiliates_import_page'
            ),
            'prettylinks' => array(
                'title' => __('Import from Pretty Links', 'linkcentral'),
                'method' => 'render_prettylinks_import_page'
            ),
        );

        // Register each import page
        foreach ($import_types as $type => $config) {
            add_submenu_page(
                'linkcentral-settings', // Non-existent parent - this makes it hidden from menu
                $config['title'],
                $config['title'],
                'linkcentral_manage_settings',
                "linkcentral-{$type}-import",
                array($this, $config['method'])
            );
        }
    }

    /**
     * Render the CSV import page.
     */
    public function render_csv_import_page() {
        // Include the dedicated CSV import page template with full functionality
        include LINKCENTRAL_PLUGIN_DIR . 'views/import-csv.php';
    }

    /**
     * Render the JSON import page.
     */
    public function render_json_import_page() {
        // Include the dedicated JSON import page template with full functionality
        include LINKCENTRAL_PLUGIN_DIR . 'views/import-json.php';
    }

    /**
     * Render the Pretty Links import page.
     */
    public function render_prettylinks_import_page() {
        // Get stats (includes availability check and caching)
        $stats = LinkCentral_PrettyLinks_Utils::get_stats();
        $is_plugin_installed = $stats['links'] > 0;

        // Set up template variables
        $import_config = array(
            'type' => 'prettylinks',
            'plugin_name' => 'Pretty Links',
            'stats' => array(
                'links' => array('label' => __('Links', 'linkcentral')),
                'categories' => array('label' => __('Categories', 'linkcentral')),
                'stats' => array('label' => __('Click Records', 'linkcentral')),
            ),
            'form' => array(),
        );

        // Always add the other options
        $import_config['form']['import_categories'] = array(
            'type' => 'checkbox',
            'label' => __('Import Categories', 'linkcentral'),
            /* translators: %s: Number of categories */
            'description' => sprintf(__('Import %s categories from Pretty Links', 'linkcentral'), '{count}'),
            'help' => __('Categories will be created in LinkCentral if they don\'t exist and assigned to the links.', 'linkcentral'),
        );

        $import_config['form']['import_stats'] = array(
            'type' => 'checkbox',
            'label' => __('Import Click Statistics', 'linkcentral'),
            /* translators: %s: Number of click records */
            'description' => sprintf(__('Import %s click records from Pretty Links', 'linkcentral'), '{count}'),
            'help' => __('This will import historical click data including IP addresses and user agents.', 'linkcentral'),
        );

        // Include the migrate import page template
        include LINKCENTRAL_PLUGIN_DIR . 'views/import-migrate.php';
    }

    /**
     * Get the import instance
     *
     * @return LinkCentral_Import The import instance
     */
    public function get_import() {
        return $this->import;
    }

}
